/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.repository;

import java.util.List;

import org.apache.commons.lang3.StringUtils;

/**
 * Updates copied pages rules
 */
public class FormPageRuleCopyUpdater extends AbstractFormCopyUpdater
{
    public void updateForm(Form originalForm, Form createdForm)
    {
        boolean hasChanges = false;
        for (FormPage originalFormPage : originalForm.getPages())
        {
            hasChanges = _updateFormPagePageRule(createdForm, originalFormPage) || hasChanges;
            for (FormQuestion originalFormQuestion : originalFormPage.getQuestions())
            {
                FormQuestion copiedQuestion = _getCopiedQuestion(createdForm, originalFormQuestion);
                hasChanges = _updateFormQuestionPageRules(createdForm, originalFormQuestion, copiedQuestion) || hasChanges;
            }
        }
        
        if (hasChanges)
        {
            createdForm.saveChanges();
        }
    }

    private boolean _updateFormPagePageRule(Form createdForm, FormPage originalFormPage)
    {
        if (originalFormPage.hasRule())
        {
            FormPageRule rule = originalFormPage.getRule();
            FormPage copiedRuleTargetPage = _getCopiedRuleTargetPage(createdForm, rule.getPageId());
            
            FormPage copiedPage = _getCopiedPage(createdForm, originalFormPage);
            copiedPage.setRule(rule.getType(), copiedRuleTargetPage != null ? copiedRuleTargetPage.getId() : null);
            return true;
        }
        return false;
    }
    
    private boolean _updateFormQuestionPageRules(Form createdForm, FormQuestion originalFormQuestion, FormQuestion copiedQuestion)
    {
        List<FormPageRule> pageRules = originalFormQuestion.getPageRules();
        for (FormPageRule rule : pageRules)
        {
            FormPage copiedTargetPage = _getCopiedRuleTargetPage(createdForm, rule.getPageId());
            copiedQuestion.addPageRules(rule.getOption(), rule.getType(), copiedTargetPage != null ? copiedTargetPage.getId() : null);
        }
        return pageRules.size() > 0;
    }
    
    private FormPage _getCopiedRuleTargetPage(Form createdForm, String originalPageId)
    {
        if (StringUtils.isNotBlank(originalPageId))
        {
            FormPage originalPage = _resolver.resolveById(originalPageId);
            return _getCopiedPage(createdForm, originalPage);
        }
        return null;
    }

    public void updateFormPage(FormPage originalFormPage, FormPage createdFormPage)
    {
        // we don't want form page rule to be updated when copying a page
    }

    public void updateFormQuestion(FormQuestion originalFormQuestion, FormQuestion createdFormQuestion)
    {
        // we don't want form page rule to be updated when copying a question
    }
    
}
