/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.commons.collections.MapUtils;

import org.ametys.cms.workflow.AmetysObjectCheckRightsCondition;
import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.cocoon.WorkflowAction;

import com.opensymphony.workflow.InvalidInputException;
import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.WorkflowException;

/**
 * Action for executing a transition on a form entry workflow.<p>
 * The following parameters are supported:
 * <dl>
 *  <dt>actionId
 *  <dd>the id of the action to fire
 * </dl>
 */
public class FormEntriesWorkflowAction extends WorkflowAction
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** Helper to retrieve the current user */
    private CurrentUserProvider _currentUserProvider;
    
    /** The users manager */
    private UserManager _userManager;
    
    /** Utility component for the handling of object represented in JSON */
    private JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    protected Map _act(Redirector redirector, Map objectModel, String source, Parameters parameters, int actionId, Map inputs) throws InvalidInputException, WorkflowException
    {
        try
        {
            String entryId = parameters.getParameter("entryId");
            FormEntry entry = _resolver.resolveById(entryId);
            
            Workflow workflow = _workflowProvider.getAmetysObjectWorkflow(entry);
            workflow.doAction(_getWorkflowId(objectModel, source, parameters), actionId, inputs);
            return _getActionResult(redirector, objectModel, source, parameters);
        }
        catch (ParameterException e)
        {
            throw new RuntimeException("Can't get entry from parameters.", e);
        }
    }
    
    @Override
    protected long _getWorkflowId(Map objectModel, String source, Parameters parameters)
    {
        try
        {
            String entryId = parameters.getParameter("entryId");
            FormEntry entry = _resolver.resolveById(entryId);
            
            return entry.getWorkflowId();
        }
        catch (Exception e)
        {
            throw new RuntimeException("Missing parameter workflowId", e);
        }
    }
    
    @Override
    protected Map<String, Object> _getInputs(Redirector redirector, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> inputs = super._getInputs(redirector, objectModel, source, parameters);
        
        String entryId = parameters.getParameter("entryId");
        FormEntry entry = _resolver.resolveById(entryId);
        
        // Provide the content key
        inputs.put(AmetysObjectCheckRightsCondition.AMETYS_OBJECT_KEY, entry);
        
        // Provide a map for providing data to the generator
        Request request = ObjectModelHelper.getRequest(objectModel);
        Map<String, Object> result = new HashMap<>();
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        inputs.put(AbstractWorkflowComponent.RESULT_MAP_KEY, result);
        
        // Set the workflow comment into the inputs.
        String comment = request.getParameter("comment");
        inputs.put("comment", comment);
        
        // Set the values for the information request function if a mail is to be sent
        String sendMailAsJSON = request.getParameter("sendMailInfo");
        Map<String, Object> sendMailInfo = _jsonUtils.convertJsonToMap(sendMailAsJSON); 
        if (MapUtils.isNotEmpty(sendMailInfo))
        {
            // Retrieve the current user name and email
            UserIdentity userIdentity = _currentUserProvider.getUser();
            User user = _userManager.getUser(userIdentity);
            if (user != null)
            {
                inputs.put(SendMailFunction.SENDER, user.getEmail());
                inputs.put(SendMailFunction.RECIPIENT, sendMailInfo.get("recipient"));
                inputs.put(SendMailFunction.SUBJECT, sendMailInfo.get("subject"));
                inputs.put(SendMailFunction.BODY, sendMailInfo.get("body"));
            }
            else
            {
                inputs.put(SendMailFunction.SEND_MAIL, "false");
            }
        }
        else
        {
            inputs.put(SendMailFunction.SEND_MAIL, "false");
        }
        
        return inputs;
    }
}
