/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

var MiniSurveyHelper = {
    
    surveyInfo: {},
    
    /**
     * Initialize the mini survey helper for a given form
     * @param {String} formId The id of the given form
     * @param {String} type The type of the chart
     * @param {Integer} width The width of the chart
     * @param {Integer} height The height of the chart
     * @param {String} ctxPath the context path
     * @param {Object} colors the colors of the chart
     * @param {String} legendColor the legend color of the chart
     * @param {String} [legendFontFamily] the legend font-family of the chart
     * @param {String} [legendFontSize=12] the legend fontsize of the chart
     * @param {String} [legendBoxWidth=40] The width of coloured box for legend
     */
    initialize: function(formId, type, width, height, ctxPath, colors, legendColor, legendFontFamily, legendFontSize, legendBoxWidth)
    {
        MiniSurveyHelper.surveyInfo[formId] = {};
        MiniSurveyHelper.surveyInfo[formId].type = type;
        MiniSurveyHelper.surveyInfo[formId].width = width;
        MiniSurveyHelper.surveyInfo[formId].height = height;
        MiniSurveyHelper.surveyInfo[formId].contextPath = ctxPath;
        MiniSurveyHelper.surveyInfo[formId].colors = colors;
        MiniSurveyHelper.surveyInfo[formId].legendColor = legendColor;
        MiniSurveyHelper.surveyInfo[formId].legendFontFamily = legendFontFamily || "'Helvetica Neue', 'Helvetica', 'Arial', sans-serif";
        MiniSurveyHelper.surveyInfo[formId].legendFontSize = Number(legendFontSize) || 12;
        MiniSurveyHelper.surveyInfo[formId].legendBoxWidth = Number(legendBoxWidth) || 40;
    },
    
    /**
     * Show chart results
     * @param {String} formId The id of the given form
     * @param {String} uniqueId The unique id
     * @param {Object} data the colors of the chart
     * @param {Boolean} answered true if the user has already answered the survey
     */
    showResult: function(formId, uniqueId, data, answered)
    {
        var form = $j('#form-' + uniqueId);
        var formInput = form.find('*[data-ametys-form]');
        var results = form.find('*[data-ametys-mini-survey-results]');
        var chart = results.find('*[data-ametys-mini-survey-chart]');
        var answered = results.find('*[data-ametys-mini-survey-answered]');
        var formQuestionTitle = form.find('*[data-ametys-mini-survey-question-title]');
        
        var nbEntries = data.nbEntries;
        
        chart.empty();
        formInput.hide();
        results.show();
        answered ? answered.show() : answered.hide();
        formQuestionTitle.show();
        
        var labels = [];
        var chartData = [];
        
        $j.each(data.questions, function (index, q)
        {
           let canvasId = "results-chart-" + uniqueId + "-" + q.id;
           chart.append('<canvas id="' + canvasId + '" width="' + MiniSurveyHelper.surveyInfo[formId].width + '" height="' + MiniSurveyHelper.surveyInfo[formId].height + '"></canvas>');
           
           var opts = q.options[0].choices;
           for (var i = 0; i < opts.length; i++)
           {
            	let value = opts[i].label;
            	let otpLabel = value;
            	if (value == '__internal_other')
                {
                	otpLabel = "{{i18n plugin.forms:PLUGINS_FORMS_DISPLAY_OTHER_OPTION_COMBOBOX}}";
                }
                else if (value == '__internal_not_answered')
                {
                	otpLabel = "{{i18n plugin.forms:PLUGINS_FORMS_STATISTICS_NOT_ANSWERED}}";
                }
                labels.push(otpLabel);
                chartData.push(Math.round(opts[i].count / nbEntries * 100));
           }
        
           var config = {
                type: MiniSurveyHelper.surveyInfo[formId].type,
                data: {
                    datasets: [{
                        label: '',
                        data: chartData,
                        backgroundColor: [
                            MiniSurveyHelper.surveyInfo[formId].colors.red,
                            MiniSurveyHelper.surveyInfo[formId].colors.orange,
                            MiniSurveyHelper.surveyInfo[formId].colors.green,
                            MiniSurveyHelper.surveyInfo[formId].colors.blue,
                            MiniSurveyHelper.surveyInfo[formId].colors.purple,
                            MiniSurveyHelper.surveyInfo[formId].colors.grey,
                            MiniSurveyHelper.surveyInfo[formId].colors.yellow
                        ]
                    }],
                    labels: labels
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                color: MiniSurveyHelper.surveyInfo[formId].legendColor,
                                font: {
                                    size: MiniSurveyHelper.surveyInfo[formId].legendFontSize,
                                    family: MiniSurveyHelper.surveyInfo[formId].legendFontFamily
                                } ,
                                boxWidth: MiniSurveyHelper.surveyInfo[formId].legendBoxWidth,
                                generateLabels(chart) {
                                    // Get the legend items
                                    const legendItems = Chart.controllers.doughnut.overrides.plugins.legend.labels.generateLabels(chart);
                                    
                                    // Return truncated legend items depending of the chart size
                                    return legendItems.map(item => {
                                        return {
                                            ...item,
                                            text: MiniSurveyHelper._truncateText(chart, item.text)
                                        };
                                    });
                                },
                            },
                            align: 'start',
                        }
                    }
                }
            };
    
            var ctx = document.getElementById(canvasId).getContext('2d');
            new Chart(ctx, config);
        });
    },
    
    /**
     * Truncate the text depending of the size of the chart. Add ... if the text is truncated
     * @param {Object} chart The chart object
     * @param {String} text The text to truncate
     */ 
    _truncateText: function(chart, text)
    {
        let ctx = chart.ctx;
        let maxWidth = chart.legend.width - chart.legend.options.labels.boxWidth - 20; // Remove 20px to be sure that the text is not truncated by the canvas

        let characters = text.split('');
        let line = "";
        for (let character of characters) 
        {
            const testLine = line ? line + character : character;
            const width = ctx.measureText(testLine).width;
            if (width > maxWidth) 
            {
                line = line.substring(0, line.length - 3).trim() + "…";
                break;
            }
            else 
            {
                line = testLine;
            }
        }

        return line;
    },
    
    /**
     * Submit the form
     * @param {String} formId The id of the given form
     * @param {String} uniqueId The unique id
     */ 
    submitForm: function(formId, uniqueId)
    {
        $j('#form-' + uniqueId + '-error').hide();
        var form = $j('#form-' + uniqueId);
        var formInput = form.find('*[data-ametys-form]');
        var errorDiv = form.find('*[data-ametys-mini-survey-error]');
        
        var values = formInput.serializeArray();
        
        var data = {};
        for(var i = 0; i < values.length; i++)
        {
            var d = data[values[i].name];
            
            if (data[values[i].name] === undefined)
            {
                data[values[i].name] = values[i].value;
            }
            else if ($j.isArray(d))
            {
                d.push(values[i].value);
            }
            else
            {
                data[values[i].name] = [d];
                data[values[i].name].push(values[i].value);
            }
        }

        $j.ajax({
            url: MiniSurveyHelper.surveyInfo[formId].contextPath + "/_plugins/forms/finish-mini-survey",
            type: 'POST',
            dataType: 'json',
            data: data,
            traditional: true,
            success: function(data) 
            {
                if (data.success)
                {
                    if (data.results.nbEntries !== undefined)
                    {
                        var info = form.find('*[data-ametys-mini-survey-info]');
                        var votesEl = info.find('*[data-ametys-mini-survey-votes]');
                        var nbVotesTpl = "{{i18n plugin.forms:PLUGINS_FORMS_SERVICE_MINI_SURVEY_NB_VOTES}}";
                        votesEl.html(data.results.nbEntries == 1 ? "{{i18n plugin.forms:PLUGINS_FORMS_SERVICE_MINI_SURVEY_ONE_VOTE}}" : nbVotesTpl.replace("{0}", data.results.nbEntries));
                        votesEl.show();
                    }
                    
                    // Display results
                    MiniSurveyHelper.showResult(formId, uniqueId, data.results, true);
                }
                else
                {
                    var errorMsg = data['error-msg'];
                    if (errorMsg)
                    {
                        errorDiv.html(errorMsg);
                    }
                    else
                    {
                        errorDiv.html("{{i18n plugin.forms:PLUGINS_FORMS_RENDER_UNAUTHORIZED}}");
                    }
                    
                    errorDiv.show();
                }
            },
            
            error: function() 
            {
                errorDiv.html("{{i18n plugin.forms:PLUGINS_FORMS_SUBMIT_ERROR}}");
                errorDiv.show();
            }
        });
        
        return false;
    },
    
    /**
     * Update the form view
     * @param {String} formId The id of the given form
     * @param {String} uniqueId The unique id
     */
    updateFormView: function(formId, uniqueId)
    {
        $j.ajax({
            url: MiniSurveyHelper.surveyInfo[formId].contextPath + "/_plugins/forms/mini-survey/dynamic-info",
            type: 'POST',
            dataType: 'json',
            data: {
                id: formId
            },
            success: function(data) 
            {
                var form = $j('#form-' + uniqueId);
                var formInput = form.find('*[data-ametys-form]');
                var results = form.find('*[data-ametys-mini-survey-results]');
                var info = form.find('*[data-ametys-mini-survey-info]');
                
                formInput.hide();
                info.hide();
                results.hide();
                
                var status = data.status ? data.status.toLowerCase() : '';
    
                if (status == 'open' || status == 'over')
                {
                    var $votesEl = info.find('*[data-ametys-mini-survey-votes]');
                    var $daysLeftEl = info.find('*[data-ametys-mini-survey-daysleft]');
                    
                    var nbVotesTpl = "{{i18n plugin.forms:PLUGINS_FORMS_SERVICE_MINI_SURVEY_NB_VOTES}}";
                    var daysLeftTpl = "{{i18n plugin.forms:PLUGINS_FORMS_SERVICE_MINI_SURVEY_DAYS_LEFT}}";
                    
                    if (data.nbEntries !== undefined)
                    {
                        $votesEl.html(data.nbEntries == 1 ? "{{i18n plugin.forms:PLUGINS_FORMS_SERVICE_MINI_SURVEY_ONE_VOTE}}" : nbVotesTpl.replace("{0}", data.nbEntries));
                        $votesEl.show();
                    }
                    
                    if (status == 'open')
                    {
                        if (data.daysLeft !== undefined)
                        {
                            $daysLeftEl.html(daysLeftTpl.replace("{0}", data.daysLeft));
                            $daysLeftEl.show();
                        }
                    }
                    else
                    {
                        $daysLeftEl.html("{{i18n plugin.forms:PLUGINS_FORMS_SERVICE_MINI_SURVEY_OVER}}");
                        $daysLeftEl.show();
                    }
                    
                    info.show();
                }
                
                if (status == 'open' && !data.answered)
                {
                    // Display form
                    formInput.show();
                }
                else if (status == 'over' || data.answered)
                {
                    // Show results
                    MiniSurveyHelper.showResult(formId, uniqueId, data.results, data.answered);
                }
                else if (status == 'coming')
                {
                    // Display nothing
                }
                
            },
            error: function() 
            {
                console.log(error);
            }
        });
                        
     }
}
