/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.generators;

import java.io.IOException;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.dao.FormQuestionDAO;
import org.ametys.plugins.forms.dao.FormQuestionDAO.FormEntryValues;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.computing.ComputingType;
import org.ametys.plugins.forms.question.types.impl.ComputedQuestionType;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * Generate the entry of a form
 * This generator is used for "my submission" service
 */
public class FormEntryInformationGenerator extends ServiceableGenerator
{
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The Ametys Object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The user manager */
    protected UserManager _userManager;

    /** The form entry DAO */
    protected FormEntryDAO _formEntryDAO;
    
    /** The form question DAO */
    protected FormQuestionDAO _formQuestionDAO;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _formEntryDAO = (FormEntryDAO) smanager.lookup(FormEntryDAO.ROLE);
        _formQuestionDAO = (FormQuestionDAO) smanager.lookup(FormQuestionDAO.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        String entryId = _getFormEntryId();
        
        contentHandler.startDocument();
        
        try
        {
            FormEntry entry = _resolver.resolveById(entryId);
            
            _checkRights(entry);
            
            _saxEntry(entry);
        }
        catch (SAXException e)
        {
            getLogger().error("An error occurred saxing entry information for entry '" + entryId + "'", e);
        }
        
        contentHandler.endDocument();
    }
    
    /**
     * Check right before saxing entry
     * @param entry the entry
     */
    protected void _checkRights(FormEntry entry)
    {
        UserIdentity currentUser = _currentUserProvider.getUser();
        if (!entry.getUser().equals(currentUser))
        {
            throw new AccessDeniedException("User '" + currentUser + "' is not allowed to access to user entry data.");
        }
    }

    /**
     * Sax the entry
     * @param entry the entry
     * @throws SAXException if a saxing error occurred
     */
    protected void _saxEntry(FormEntry entry) throws SAXException
    {
        AttributesImpl formAttrs = new AttributesImpl();
        formAttrs.addCDATAAttribute("formId", entry.getForm().getId());
        formAttrs.addCDATAAttribute("label", entry.getForm().getTitle());
        formAttrs.addCDATAAttribute("entryId", String.valueOf(entry.getEntryId()));
        
        ZonedDateTime submittedAt = entry.getSubmitDate().toInstant().atZone(ZoneId.systemDefault());
        formAttrs.addCDATAAttribute("id", entry.getId());
        formAttrs.addCDATAAttribute("creationDate", DateUtils.zonedDateTimeToString(submittedAt));
        UserIdentity userIdentity = entry.getUser();
        if (userIdentity != null)
        {
            formAttrs.addCDATAAttribute("userId", UserIdentity.userIdentityToString(userIdentity));
            
            User user = _userManager.getUser(userIdentity);
            if (user != null)
            {
                formAttrs.addCDATAAttribute("user", user.getFullName());
            }
        }
        
        Long entryId = entry.getEntryId();
        if (entryId != null)
        {
            formAttrs.addCDATAAttribute("entryId", String.valueOf(entryId));
        }
        
        _addAdditionalEntryAttributes(entry, formAttrs);
        
        XMLUtils.startElement(contentHandler, "entry", formAttrs);
        
        Optional<Long> currentStepId = entry.getForm().hasWorkflow() ? Optional.of(_formEntryDAO.getCurrentStepId(entry)) : Optional.empty();
        List<FormQuestion> questions = _getQuestions(entry, currentStepId).stream()
                .filter(this::_displayField)
                .collect(Collectors.toList());
        for (FormQuestion question : questions)
        {
            _saxQuestion(question, entry);
        }
        
        XMLUtils.endElement(contentHandler, "entry");
    }
    
    /**
     * Get the questions of the given entry
     * @param entry the entry
     * @param currentStepId the current step id. Can be empty
     * @return the list of question
     */
    protected List<FormQuestion> _getQuestions(FormEntry entry, Optional<Long> currentStepId)
    {
        UserIdentity currentUser = _currentUserProvider.getUser();
        
        // Send all information if the current user is not null and not equal to the entry user
        boolean onlyReadableQuestion = currentUser != null && Objects.equals(currentUser, entry.getUser());
        return _formQuestionDAO.getRuleFilteredQuestions(entry.getForm(), new FormEntryValues(null, entry), currentStepId, false, onlyReadableQuestion);
    }
    
    /**
     * Sax the question
     * @param question the question
     * @param entry the entry
     * @throws SAXException if a saxing exception occurred
     */
    protected void _saxQuestion(FormQuestion question, FormEntry entry) throws SAXException
    {
        FormQuestionType type = question.getType();
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("type", type.getStorageType(question));
        attrs.addCDATAAttribute("typeId", type.getId());
        attrs.addCDATAAttribute("label", question.getTitle());
        attrs.addCDATAAttribute("name", question.getNameForForm());
        
        XMLUtils.startElement(contentHandler, "field", attrs);
        type.saxEntryValue(contentHandler, question, entry);
        XMLUtils.endElement(contentHandler, "field");
    }
    
    /**
     * Add additional entry attributes
     * @param entry the entry
     * @param attrs the attributes
     */
    protected void _addAdditionalEntryAttributes(FormEntry entry, AttributesImpl attrs)
    {
        // Do nothing
    }

    /**
     * <code>true</code> if the field can be displayed
     * @param question the question
     * @return <code>true</code> if the field can be displayed
     */
    protected boolean _displayField(FormQuestion question)
    {
        FormQuestionType type = question.getType();
        // This generator is used for front display or mail. Do not display computed field with only server computed value
        if (type instanceof ComputedQuestionType questionType)
        {
            ComputingType computingType = questionType.getComputingType(question);
            Set<String> displayXSLTs = questionType.getDisplayXSLTs();
            return displayXSLTs != null && !displayXSLTs.isEmpty() && computingType.hasComputedValue();
        }
        return true;
    }
    
    private String _getFormEntryId()
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        // Search in the request
        String entryId = (String) request.get("entryId");
        
        // Search in the parent context
        if (entryId == null)
        {
            @SuppressWarnings("unchecked")
            Map<String, Object> params = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
            if (params != null)
            {
                entryId = (String) params.get("entryId");
            }
            
            // Search in the request attributes
            if (entryId == null)
            {
                entryId = (String) request.getAttribute("entryId");
            }
        }
        
        return entryId;
    }
}
