/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.File;
import org.ametys.cms.data.holder.DataHolderRelativeDisableCondition;
import org.ametys.cms.data.holder.DataHolderRelativeDisableConditions;
import org.ametys.cms.data.holder.DataHolderRelativeDisableConditionsHelper;
import org.ametys.plugins.forms.helper.FormWorkflowHelper;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.ViewItemGroup;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.DefaultValidator;
import org.ametys.runtime.parameter.Validator;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * Default abstract class to represent a question type.
 * This default implementation allow to define an illustration, a description and mark the question as mandatory
 */
public abstract class AbstractFormQuestionType extends AbstractStaticFormQuestionType implements IllustrableAwareQuestionType, DescriptibleAwareQuestionType, MandatoryAwareQuestionType, ConfidentialAwareQuestionType, RestrictiveAwareQuestionType
{
    /** The workflow helper component */
    protected WorkflowHelper _workflowHelper;
    
    /** The data holder relative disable condition helper */
    protected DataHolderRelativeDisableConditionsHelper _disableConditionsHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _workflowHelper = (WorkflowHelper) manager.lookup(WorkflowHelper.ROLE);
        _disableConditionsHelper = (DataHolderRelativeDisableConditionsHelper) manager.lookup(DataHolderRelativeDisableConditionsHelper.ROLE);
    }
    
    public List<ModelItem> getRestrictiveModelItems()
    {
        ElementDefinition readingCheckbox = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_READING_CHECKBOX, ModelItemTypeConstants.BOOLEAN_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_READING_CHECKBOX", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_READING_CHECKBOX_DESC", null);
        ElementDefinition<Long> reading = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_READING, ModelItemTypeConstants.LONG_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_READING", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_READING_DESC", null);
        reading.setMultiple(true);
        
        Map<String, I18nizableText> readingWidgetParameters  = new HashMap<>();
        readingWidgetParameters.put("naturalOrder", new I18nizableText("true"));
        reading.setWidgetParameters(readingWidgetParameters);
        Map<String, I18nizableText> widgetParameters  = new HashMap<>();
        widgetParameters.put("emptyText", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_READING_PLACEHOLDER"));
        reading.setWidgetParameters(widgetParameters);
        
        DisableConditions disableConditions = new DataHolderRelativeDisableConditions();
        DisableCondition condition = new DataHolderRelativeDisableCondition(ATTRIBUTE_READING_CHECKBOX, OPERATOR.NEQ, "true", _disableConditionsHelper);
        disableConditions.getConditions().add(condition);
        reading.setDisableConditions(disableConditions);
        
        ElementDefinition writingCheckbox = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_WRITING_CHECKBOX, ModelItemTypeConstants.BOOLEAN_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_WRITING_CHECKBOX", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_WRITING_CHECKBOX_DESC", null);
        ElementDefinition<Long> writing = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_WRITING, ModelItemTypeConstants.LONG_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_WRITING", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_WRITING_DESC", new DefaultValidator(null, true));
        writing.setMultiple(true);
        
        Map<String, I18nizableText> writingWidgetParameters  = new HashMap<>();
        writingWidgetParameters.put("naturalOrder", new I18nizableText("true"));
        writing.setWidgetParameters(writingWidgetParameters);

        DisableConditions writingDisableConditions = new DataHolderRelativeDisableConditions();
        DisableCondition writingCondition = new DataHolderRelativeDisableCondition(ATTRIBUTE_WRITING_CHECKBOX, OPERATOR.NEQ, "true", _disableConditionsHelper);
        writingDisableConditions.getConditions().add(writingCondition);
        writing.setDisableConditions(writingDisableConditions);
        
        return List.of(readingCheckbox, reading, writingCheckbox, writing);
    }
    
    @Override
    protected List<ModelItem> _getModelItems()
    {
        List<ModelItem> modelItems = super._getModelItems();
        
        //DESCRIPTION
        modelItems.add(getDescriptionModelItem());

        //RESTRICTION BY WORKFLOW
        modelItems.addAll(getRestrictiveModelItems());
        
        //ILLUSTRATION & ALTERNATE TEXT
        modelItems.addAll(getIllustrationModelItems());
        
        //MANDATORY
        modelItems.add(getMandatoryModelItem());
        
        //CONFIDENTIALITY
        modelItems.add(getConfidentialityModelItem());
        
        return modelItems;
    }
    
    @Override
    protected List<ViewItemGroup> _getTabs(Form form)
    {
        if (StringUtils.isEmpty(form.getWorkflowName()))
        {
            return List.of(
                    _getMainTab(),
                    _getAdvancedTab(),
                    _getRulesTab(),
                    getIllustrationTab(getModel())
                    );
        }
        else
        {
            return List.of(
                    _getMainTab(),
                    _getAdvancedTab(),
                    _getRulesTab(),
                    getRestrictiveTab(getModel(), form),
                    getIllustrationTab(getModel())
                    );
        }
    }
    
    /**
     * Define the content of the main tab
     * @return the main tab definition
     */
    protected SimpleViewItemGroup _getMainTab()
    {
        SimpleViewItemGroup fieldset = super._createMainTab();
        
        ViewElement title = new ViewElement();
        title.setDefinition((ElementDefinition< ? >) getModel().getModelItem(FormQuestion.ATTRIBUTE_TITLE));
        fieldset.addViewItem(title);
        
        fieldset.addViewItem(getDescriptionViewElement(getModel()));
        
        fieldset.addViewItem(getMandatoryViewElement(getModel()));
        
        return fieldset;
    }
    
    /**
     * Define the content of the advanced tab
     * @return the advanced tab definition
     */
    protected SimpleViewItemGroup _getAdvancedTab()
    {
        SimpleViewItemGroup advancedFieldset = super._createAdvancedTab();
        
        ViewElement nameForForms = new ViewElement();
        nameForForms.setDefinition((ElementDefinition< ? >) getModel().getModelItem(FormQuestion.ATTRIBUTE_NAME_FOR_FORM));
        advancedFieldset.addViewItem(nameForForms);
        
        advancedFieldset.addViewItem(getConfidentialityViewElement(getModel()));
        
        return advancedFieldset;
    }
    
    @Override
    protected ModelItem _getEntryModelItem(FormQuestion question)
    {
        // Add the mandatory validator to the entry model item
        ModelItem item = super._getEntryModelItem(question);
        ((ElementDefinition) item).setValidator(getMandatoryValidator(question));
        return item;
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished(FormQuestion question)
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished(question);
        fieldNames.add(MandatoryAwareQuestionType.ATTRIBUTE_MANDATORY);
        fieldNames.add(RestrictiveAwareQuestionType.ATTRIBUTE_READING_CHECKBOX);
        fieldNames.add(RestrictiveAwareQuestionType.ATTRIBUTE_READING);
        return fieldNames;
    }
    
    public WorkflowDescriptor getWorkflowDescriptor(Form form)
    {
        return _workflowHelper.getWorkflowDescriptor(form.getWorkflowName());
    }

    public SimpleViewItemGroup getRestrictiveTab(Model model, Form form)
    {
        SimpleViewItemGroup restrictiveFieldset = new SimpleViewItemGroup();
        restrictiveFieldset.setName("restriction");
        restrictiveFieldset.setLabel(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_FIELDSET"));
        restrictiveFieldset.setDescription(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_FIELDSET_DESCRIPTION"));
        restrictiveFieldset.setRole(ViewItemGroup.TAB_ROLE);

        ViewElement readingCheckbox = new ViewElement();
        readingCheckbox.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_READING_CHECKBOX));
        restrictiveFieldset.addViewItem(readingCheckbox);
        
        ViewElement reading = new ViewElement();
        @SuppressWarnings("unchecked")
        ElementDefinition<Long> readingElementDefinition = (ElementDefinition<Long>) model.getModelItem(ATTRIBUTE_READING);
        readingElementDefinition.setEnumerator(_getEnumerator(form, true));
        reading.setDefinition(readingElementDefinition);

        restrictiveFieldset.addViewItem(reading);

        ViewElement writingCheckbox = new ViewElement();
        writingCheckbox.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_WRITING_CHECKBOX));
        restrictiveFieldset.addViewItem(writingCheckbox);

        ViewElement writing = new ViewElement();
        @SuppressWarnings("unchecked")
        ElementDefinition<Long> writingElementDefinition = (ElementDefinition<Long>) model.getModelItem(ATTRIBUTE_WRITING);
        writingElementDefinition.setEnumerator(_getEnumerator(form, false));
        writing.setDefinition(writingElementDefinition);
        
        restrictiveFieldset.addViewItem(writing);
        
        return restrictiveFieldset;
    }
    
    private StaticEnumerator<Long> _getEnumerator(Form form, boolean forReading)
    {
        StaticEnumerator<Long> extensionEnum = new StaticEnumerator<>();
        
        WorkflowDescriptor workflowDesc = getWorkflowDescriptor(form);
        List<StepDescriptor> steps = workflowDesc.getSteps();

        for (StepDescriptor step : steps)
        {
            if (forReading || _hasEditAction(step))
            {
                extensionEnum.add(new I18nizableText("plugin.forms", step.getName()), Long.valueOf(step.getId()));
            }
        }
        
        return extensionEnum;
    }
    
    @SuppressWarnings("unchecked")
    private boolean _hasEditAction(StepDescriptor step)
    {
        List<ActionDescriptor> actions = step.getActions();
        return actions.stream()
                .map(ActionDescriptor::getMetaAttributes)
                .filter(m -> FormWorkflowHelper.EDIT_ACTION.equals(m.getOrDefault("action-type", null)))
                .findAny()
                .isPresent();
    }

    public boolean isReadRestricted(FormQuestion question)
    {
        return question.getValue(ATTRIBUTE_READING_CHECKBOX, false, false);
    }

    public boolean isModifiable(FormQuestion question)
    {
        return question.getValue(ATTRIBUTE_WRITING_CHECKBOX, false, false);
    }

    public List<Long> getReadingSteps(FormQuestion question)
    {
        if (question.hasValue(ATTRIBUTE_READING))
        {
            return Arrays.asList(question.getValue(ATTRIBUTE_READING));
        }
        else
        {
            return List.of();
        }
    }

    public List<Long> getWritingSteps(FormQuestion question)
    {
        if (question.hasValue(ATTRIBUTE_WRITING))
        {
            return ListUtils.sum(Arrays.asList(question.getValue(ATTRIBUTE_WRITING)), List.of(INITIAL_WORKFLOW_ID));
        }
        else
        {
            return List.of(INITIAL_WORKFLOW_ID);
        }
    }

    public ModelItem getDescriptionModelItem()
    {
        return _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_DESCRIPTION, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_DESCRIPTION", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_DESCRIPTION_DESC", null);
    }
    
    public ViewElement getDescriptionViewElement(Model model)
    {
        ViewElement description = new ViewElement();
        description.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_DESCRIPTION));
        return description;
    }
    
    public String getDescription(FormQuestion question)
    {
        return question.getValue(DescriptibleAwareQuestionType.ATTRIBUTE_DESCRIPTION);
    }

    public List<ModelItem> getIllustrationModelItems()
    {
        ElementDefinition illustration = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_PICTURE, org.ametys.cms.data.type.ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_IMAGE", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_IMAGE_DESC", null);
        illustration.setWidget("edition.file");
        Map<String, I18nizableText> widgetParams = new HashMap<>();
        widgetParams.put("allowSources", new I18nizableText("external"));
        widgetParams.put("filter", new I18nizableText("image"));
        illustration.setWidgetParameters(widgetParams);
        
        ElementDefinition illustrationAlternative = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_PICTURE_ALTERNATIVE, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_IMAGE_ALT", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_IMAGE_ALT_DESC", null);
        
        return List.of(illustration, illustrationAlternative);
    }
    
    public SimpleViewItemGroup getIllustrationTab(Model model)
    {
        SimpleViewItemGroup pictureFieldset = new SimpleViewItemGroup();
        pictureFieldset.setName("illustration");
        pictureFieldset.setLabel(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_IMAGE_FIELDSET"));
        pictureFieldset.setRole(ViewItemGroup.TAB_ROLE);
        
        ViewElement picture = new ViewElement();
        picture.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_PICTURE));
        pictureFieldset.addViewItem(picture);
        
        ViewElement pictureAlternative = new ViewElement();
        pictureAlternative.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_PICTURE_ALTERNATIVE));
        pictureFieldset.addViewItem(pictureAlternative);
        
        return pictureFieldset;
    }
    
    public File getIllustration(FormQuestion question)
    {
        return question.getValue(ATTRIBUTE_PICTURE);
    }
    
    public String getIllustrationAlternative(FormQuestion question)
    {
        return question.getValue(ATTRIBUTE_PICTURE_ALTERNATIVE);
    }
    
    public ModelItem getMandatoryModelItem()
    {
        return _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_MANDATORY, ModelItemTypeConstants.BOOLEAN_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_MANDATORY", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_MANDATORY_DESC", null);
    }
    
    public ViewElement getMandatoryViewElement(Model model)
    {
        ViewElement mandatory = new ViewElement();
        mandatory.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_MANDATORY));
        return mandatory;
    }
    
    public Validator getMandatoryValidator(FormQuestion question)
    {
        return new DefaultValidator(null, isMandatory(question));
    }
    
    public boolean isMandatory(FormQuestion question)
    {
        return question.getValue(ATTRIBUTE_MANDATORY, false, false);
    }
    
    public ModelItem getConfidentialityModelItem()
    {
        return _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_CONFIDENTIALITY, ModelItemTypeConstants.BOOLEAN_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_CONFIDENTIAL", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_CONFIDENTIAL_DESC", null);
    }
    
    public ViewItem getConfidentialityViewElement(Model model)
    {
        ViewElement confidential = new ViewElement();
        confidential.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_CONFIDENTIALITY));
        
        return confidential;
    }

    public boolean isConfidential(FormQuestion formQuestion)
    {
        return formQuestion.getValue(ATTRIBUTE_CONFIDENTIALITY, false, false);
    }
}
