/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The static class for edition from front office
 */
AmetysFrontEdition.comm = {
    /**
     * @private
     * Get the current server informations and value of the given metadata
     * @param {String} contentId The content identifier
     * @param {String[]} metadataPaths The metadata paths
     * @param {Number[]} workflowIds Workflow Ids to check
     * @param {Function} callback The function called when informations are available
     * @param {Object} callback.data The metadatas data
     * @param {String} callback.data.[metadataname].label The label
     * @param {String} callback.data.[metadataname].description The description
     * @param {Boolean} callback.data.[metadataname].multiple Is the metadata multiple?
     * @param {String} callback.data.[metadataname].type The metadata type, such as "TYPE"
     * @param {Object} callback.data.[metadataname].value The current value
     * @param {String} callback.data.[metadataname].widget The required back-office widget
     * @param {Object} callback.data.[metadataname].widget-params The parameters for the widget
     */
    getServerInfos: function(contentId, metadataPaths, workflowIds, callback)
    {
        var metadataPathsAsString = metadataPaths.join(';');
        var workflowIdsPathsAsString = workflowIds.join(';');
        Ametys.data.ServerComm.send({
            plugin: 'front-edition',
            url: 'content/get-server-values.json',
            parameters: {
                "contentId": contentId,
                "workflowIds": workflowIdsPathsAsString,
                "isEditionMetadataSet": true,
                "metadataPaths": metadataPathsAsString,
                "validateContent": true
            },
            responseType: 'text',
            waitMessage: "{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_LOADING}}",
            callback: {
                ignoreOnError: false, 
                handler: function(data) {
                    if (data == null)
                    {
                        console.error("An error occurred while retrieving content " + contentId);
                        callback(null);
                    }
                    else
                    {
                        var response =  Ext.decode(data.firstChild.textContent, true);
                        callback(response);
                    }
                }
            }
        });
    },
    /**
     * @private
     * Save the metadata value
     * @param {Number} editActionId the edit worflow action id
     * @param {String} contentId The content identifier
     * @param {String} metadataPath The metadata path
     * @param {String} value The value to set
     * @param {Function} callback The function called when informations are available
     * @param {Object} callback.data The metadata data
     * @param {String} callback.data.label The label
     * @param {String} callback.data.description The description
     * @param {Boolean} callback.data.multiple Is the metadata multiple?
     * @param {String} callback.data.type The metadata type, such as "TYPE"
     * @param {Object} callback.data.value The current value
     * @param {String} callback.data.widget The required back-office widget
     * @param {Object} callback.data.widget-params The parameters for the widget
     */
    save: function(editActionId, contentId, metadataPath, value, callback)
    {
        var values = {};
        values[metadataPath] = value;
        AmetysFrontEdition.comm.saveMulti(editActionId, contentId, values, false, callback);
    },
    /**
     * Save multiple metadata at once
     * @param {Number} editActionId workflow action id
     * @param {String} contentId content id
     * @param {Object} datas metadatas
     * @param {Boolean} ignoreWarning true to ignore warning
     * @param {Function} callback callback function
     * @param {Object} callbackParameters callback function parameters (after the ajax result)
     */
    saveMulti: function(editActionId, contentId, datas, ignoreWarning, callback, callbackParameters)
    {
        var values = {};
        for (var key in datas)
        {
            if (datas.hasOwnProperty(key))
            {
                values["content.input." + key] = datas[key];
            }
        }
        var parameters = {
            "contentId": contentId,
            "quit": true,
            "values": values,
            "ignore.warnings": ignoreWarning
        };
        
        callbackParameters = callbackParameters || {};
        callbackParameters.args = {
            actionId: editActionId,
            contentId: contentId,
            datas: datas
        };
        
        AmetysFrontEdition.comm.doAction(editActionId, parameters, callback, callbackParameters);
    },
    /**
     * Execute a workflow action
     * @param {Number} actionId workflow action id
     * @param {Object} parameters action parameters
     * @param {Function} callback callback
     * @param {Object} callbackParameters callback parameters (after the ajax result)
     */
    doAction: function(actionId, parameters, callback, callbackParameters)
    {
        callback = callback || function () {};
        
        Ametys.data.ServerComm.send({
            plugin: 'cms',
            url: 'do-action/' + actionId,
            parameters: parameters,
            responseType: 'xml',
            waitMessage: "{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_SAVING}}",
            callback: {
                ignoreOnError: false, 
                handler: function(data) {
                    Ext.getBody().unmask();

                    if (data == null)
                    {
                        console.error("An error occurred while doing action " + actionId);
                    }
                    callback(data, callbackParameters);
                }
            }
        });
    },
    
    /**
     * Delete a content
     * @param {String} contentId the content id
     * @param {Function} callback callback
     * @param {Object} callbackParameters callback parameters (after the ajax result)
     */
    deleteContent: function(contentId, callback, callbackParameters)
    {
        callback = callback || function () {};
        
        Ametys.data.ServerComm.callMethod({
            role: "org.ametys.web.repository.content.WebContentDAO",
            methodName: "trashContents",
            parameters: [[contentId]],
            callback: {
                scope: this,
                handler: callback,
                arguments: callbackParameters
            },
            waitMessage: true,
            errorMessage: true
        });
    },
    
    /**
     * Add tags to a content
     * @param {String} contentId the content id
     * @param {String[]} tags The tags to add
     * @param {Function} callback callback
     * @param {Object} callbackParameters callback parameters (after the ajax result)
     */
    addTags: function(contentId, tags, callback, callbackParameters)
    {
        callback = callback || function () {};
        
        Ametys.data.ServerComm.callMethod({
            role: "org.ametys.cms.repository.ContentDAO",
            methodName: "tag",
            parameters: [[contentId], tags, 'INSERT', Ametys.getAppParameters() || {}],
            callback: {
                scope: this,
                handler: callback,
                arguments: callbackParameters
            },
            waitMessage: true,
            errorMessage: true
        });
    },
    
    /**
     * Remove tags on a content
     * @param {String} contentId the content id
     * @param {String[]} tags The tags to remove
     * @param {Function} callback callback
     * @param {Object} callbackParameters callback parameters (after the ajax result)
     */
    removeTags: function(contentId, tags, callback, callbackParameters)
    {
        callback = callback || function () {};
        
        Ametys.data.ServerComm.callMethod({
            role: "org.ametys.cms.repository.ContentDAO",
            methodName: "tag",
            parameters: [[contentId], tags, 'REMOVE', Ametys.getAppParameters() || {}],
            callback: {
                scope: this,
                handler: callback,
                arguments: callbackParameters
            },
            waitMessage: true,
            errorMessage: true
        });
    },
    
    /**
     * Get the name of a workflow action (if this workflow action is available for a content)
     * @param {String} contentId content on which the action will be applied
     * @param {Number[]} workflowActionIds array of workflow action id
     * @param {Function} callback function called with the result as a map {id:name, id2:name2...}
     */
    getWorkflowNames: function(contentId, workflowActionIds, callback)
    {
        if (workflowActionIds.length > 0)
        {
            Ametys.data.ServerComm.callMethod({
                role: "org.ametys.plugins.frontedition.FrontEditionHelper",
                methodName: "getWorkflowActionName",
                parameters: [contentId, "content", workflowActionIds],
                callback: {
                    scope: this,
                    handler: AmetysFrontEdition.comm._getWorkflowNamesCb,
                    arguments: callback
                },
                errorMessage: true
            });
        }
        else
        {
            AmetysFrontEdition.comm._getWorkflowNamesCb(null, callback);
        }
    },
    _getWorkflowNamesCb : function(workflowNames, callback)
    {
        if (Ext.isFunction(callback))
        {
            callback(workflowNames);
        }
    },
    /**
     * Lock or UnLock contents
     * @param {String[]} contentIds ids of contents to lock/unlock
     * @param {String} mode "unlock" or "lock"
     */
    unlockOrLock : function(contentIds, mode, callback)
    {
        if (!Ext.isArray(contentIds))
        {
            contentIds = [contentIds];
        }
        Ametys.data.ServerComm.callMethod({
                role: "org.ametys.cms.lock.LockContentManager",
                methodName: "unlockOrLock",
                parameters: [contentIds, mode],
                callback: callback,
                errorMessage: true
            });
    }
};
