/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The static class for front edition toolbar for contents
 */
AmetysFrontEditionContentToolbar = {
    
    /**
     * @readonly
     * @property {String} MENU_CLASSNAME The css classname for toolbar's menus
     */
    MENU_CLASSNAME: 'ametys-front-edition-content-toolbar-menu',
    
    /**
     * @readonly
     * @property {String} ACTION_BUTTON_CLASSNAMES The css classnames separated by coma for toolbar's action buttons
     */
    ACTION_BUTTON_CLASSNAMES: 'ametys-front-edition-content-toolbar-btn ametys-front-edition-content-toolbar-btn-action',
    
    /**
     * @readonly
     * @property {String} STATUS_BUTTON_CLASSNAMES The css classnames separated by coma for toolbar's status buttons
     */
    STATUS_BUTTON_CLASSNAMES: 'ametys-front-edition-content-toolbar-btn ametys-front-edition-content-toolbar-btn-status',
    
    /**
     * @readonly
     * @property {String} STATUS_NEG_BUTTON_CLASSNAME Additional css classname for toolbar's status buttons of type 'negative'
     */
    STATUS_NEG_BUTTON_CLASSNAME: 'negative',
    
    /**
     * @readonly
     * @property {String} STATUS_NEG_BUTTON_CLASSNAME Additional css classname for toolbar's status buttons of type 'positive'
     */
    STATUS_POS_BUTTON_CLASSNAME: 'positive',
    
    /**
     * @readonly
     * @property {String} STATUS_NEG_BUTTON_CLASSNAME Additional css classname for toolbar's status buttons of type 'neutral' (default)
     */
    STATUS_NEUTRAL_BUTTON_CLASSNAME: 'neutral',
    
    /**
     * Create the content toolbar
     * @param {String} selector The jQuery selector for toolbar
     * @param {Object} config The toolbar configuration
     * @param {Object} config.label The label of toolbar
     * @param {Object[]} config.items The toolbar items and their organization (ex: ['status-history', 'current-step',  {type: 'menu', items: ['workflow-actions']}, {type: 'menu', label: 'More...', items: ['edit','tag', {type: 'menu', label: "Move", items: ['move-up', 'move-down']}, 'remove']])
     * @param {Object} config.actionStatus The active status of each action depending on user rights and/or content's workflow and/or position of content in the page (ex: {'edit': true, 'tag': true, 'move': false, 'remove': false})
     * @param {Object} config.workflowActions The available workflow actions (restricted to the requested workflow actions)
     * @param {Number[]} config.availableWorkflowActions The id of all available workflow actions for current user (whatever the toolbar configuration)
     * @param {Number[]} config.excludedWorkflowActionsOutsideEditionMode The id of workflow actions to exclude if not in edition mode
     * @param {Object} config.step The current step
     * @param {Object} config.workflowSteps The workflow steps configuration 
     * @param {String} config.contentId The content id
     * @param {String} config.zoneItemId The zone item id
     * @param {String} config.zoneName The zone name
     * @param {String} [config.editViewName] The view name to use for edition
     */
    createContentToolbar: function(selector, config)
    {
        var $toolbar = $j(selector);
        $toolbar.attr("role", "toolbar");
        if (config.label)
        {
            $toolbar.attr("aria-label", config.label);
        }
        
         var items = config.items || []; // ex: items = ['status-history', 'current-step',  {type: 'menu', items: ['workflow-actions']}, {type: 'menu', label: 'More...', items: ['edit','tag', {type: 'menu', label: "Move", items: ['move-up', 'move-down']}, 'remove']]
         var actionStatus = config.actionStatus || {}; // ex: actionStatus = {'edit': true, 'tag': true, 'move': false, 'remove': false}
         
         for (var i=0; i < items.length; i++)
         {
	           if (typeof items[i] == 'string')
	           {
	               items[i] = {type: items[i]};
	           }
               
               var type = items[i].type;
               if (type == 'menu')
               {
                    var menuItems = AmetysFrontEditionContentToolbar.getContentMenuItems(config, items[i].items);
                    if (menuItems && Object.keys(menuItems).length > 0)
                    {
                        var id = 'ametys-front-edition-content-toolbar-btn-' + type + '-' + Math.random().toString(16).slice(-8);
                        if (Object.keys(menuItems).length == 1)
                        {
                            // transform unique menu item into button
                            var itemType = Object.keys(menuItems)[0];
                            var menuItem = menuItems[itemType];
                            var $btn = AmetysFrontEditionContentToolbar._getActionToolbarButton({iconCls: menuItem.icon, label: menuItem.name, handler: menuItem.callback, type: menuItem.type, id: id});
                            $btn.appendTo($toolbar);
                            
                            if (menuItem.items)
                            {
                                // attach menu to button
                                AmetysFrontEditionContentToolbar._createMenu('#' + id, menuItem.items, false);
                            }
                        }
                        else
                        {
                            // create menu button
                            var label = items[i].hasOwnProperty('label') ? items[i].label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_ACTIONS_MENU_LABEL}}";
	                        var $btn = AmetysFrontEditionContentToolbar._getActionToolbarButton({
                                iconCls: items[i].hasOwnProperty('icon') ? items[i].icon : 'fas fa-chevron-down', 
                                label: label,
                                type: type,
                                id: id,
                                hideLabel: items[i].hideLabel
                            });
                            $btn.appendTo($toolbar);
                            
                            AmetysFrontEditionContentToolbar._createMenu('#' + id, menuItems, true);
                        }
                    }
               }
               else if (type == 'current-step' && AmetysFrontEdition.EDITION_MODE) // current step can be displayed only in edition mode
               {
                    var $btn = AmetysFrontEditionContentToolbar._getStatusToolbarButton(config.step.id, config.step.label, config.workflowSteps || {})
                    $btn.appendTo($toolbar);
               }
               else if (type == 'status-history' && AmetysFrontEdition.EDITION_MODE) // status history can be displayed only in edition mode
               {
                    AmetysFrontEditionContentToolbar._addHistoryStatusToolbarButtons($toolbar, config.statusHistory, config.workflowSteps || {})
               }
               else if (type == 'workflow-actions')
               {
                    // insert all available actions as buttons
                    AmetysFrontEditionContentToolbar._addWorkflowActionsButtons($toolbar, items[i], config);
               }
               else if (type.indexOf('workflow-action-') == 0)
	           {
                    // insert specific action if available as button
                    var actionId = type.substring('workflow-action-'.length);
	                var $btn = AmetysFrontEditionContentToolbar._getWorkflowActionButton(Number(actionId), config, items[i]);
                    if ($btn)
                    {
                        $btn.appendTo($toolbar);
                    }
	           }
               else if (actionStatus[type] && AmetysFrontEditionContentToolbar.CONTENT_ACTIONS_BUTTON[type] || (type == 'edit' && config.availableWorkflowActions && config.availableWorkflowActions.indexOf(AmetysFrontEdition.editActionId) != -1) /* check edition action id is part of available workflow actions */)
	           {
                     var $btn = AmetysFrontEditionContentToolbar.CONTENT_ACTIONS_BUTTON[type](config, items[i]);
                     $btn.appendTo($toolbar);
	           }
               else if (!AmetysFrontEditionContentToolbar.CONTENT_ACTIONS_BUTTON[type])
               {
                    console.warn('[Front edition toolbar] No content action available for type ' + type);
               }
        }
        
        // Managing focus within toolbar using a roving tabindex
        AmetysFrontEditionToolbarHelper.handleKeyBoardNavigation($toolbar.get(0), false);
    },
    
     /**
     * @private
     * Create a menu
     * @param {String} selector The jQuery selector matching the button to trigger on
     * @param {Object} menuItems The menu items
     * @param {boolean} triggerActive true to trigger active
     */
    _createMenu: function(selector, menuItems, triggerActive)
    {
        AmetysFrontEditionToolbarHelper.attachContextMenu({
            selector: selector,
            menuItems: menuItems,
            className: AmetysFrontEditionPageToolbar.MENU_CLASSNAME,
            triggerActive: triggerActive
        });
    },
    
    /**
     * Get the menu items for a content
     * @param {Object} config The toolbar configuration
     * @param {Object[]} items The items of menu
     * @return {Object} the menu items
     */
    getContentMenuItems: function (config, items)
    {
        var menuItems = {};
        var hasItems = false;
        
        var items = items || [];
        var actionStatus = config.actionStatus || {};
        
        for (var i=0; i < items.length; i++)
        {
            if (typeof items[i] == 'string')
            {
                items[i] = {type: items[i]};
            }
            
            var type = items[i].type;
            if (type == 'workflow-actions')
            {
                // insert all available workflow actions as menu items 
                hasItems = AmetysFrontEditionContentToolbar._addWorkflowActionsItems(menuItems, config, items[i]) || hasItems;
            }
            else if (type.indexOf('workflow-action-') == 0)
            {
                // insert specific workflow actions if available as menu item 
                var actionId = type.substring('workflow-action-'.length);
                hasItems = AmetysFrontEditionContentToolbar._addWorkflowActionItem(Number(actionId), menuItems, config, items[i]) || hasItems;
            }
            else if (type == 'menu')
            {
                // sub menu
                var subMenuItems = AmetysFrontEditionContentToolbar.getContentMenuItems(config, items[i].items);
                if (subMenuItems && Object.keys(subMenuItems).length > 0)
                {
	                var subMenuSize = Object.keys(subMenuItems).length;
	                if (subMenuSize == 1)
	                {
	                    // transform unique sub menu item to menu item
	                    var key = Object.keys(subMenuItems)[0];
	                    menuItems[key] = subMenuItems[key];
	                    hasItems = true;
	                }
	                else if (subMenuSize > 1)
	                {
	                    menuItems['menu-' + i] = {
	                        name: items[i].hasOwnProperty('label') ? items[i].label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_ACTIONS_MENU_LABEL}}", 
	                        icon: items[i].icon,
	                        items: subMenuItems
	                    }
	                    hasItems = true;
	                }
                }
            }
            else if (actionStatus[type] && AmetysFrontEditionContentToolbar.CONTENT_ACTIONS_MENU_ITEM[type] || (type == 'edit' && config.availableWorkflowActions && config.availableWorkflowActions.indexOf(AmetysFrontEdition.editActionId) != -1))
            {
                menuItems[type] = AmetysFrontEditionContentToolbar.CONTENT_ACTIONS_MENU_ITEM[type](config, items[i]);
                hasItems = true;
            }
            else if (!AmetysFrontEditionContentToolbar.CONTENT_ACTIONS_MENU_ITEM[type])
            {
                console.warn('[Front edition toolbar] No item available for type ' + type);
            }
        }
        
        return hasItems ? menuItems : null; // return null if no action is available
    },
    
    /**
     * @private
     * Add available workflow buttons to toolbar
     * @param {Object} items The menu items
     * @param {Object} config The toolbar configuration
     * @return {Boolean} true if at least one item was added
     */
    _addWorkflowActionsButtons: function (toolbar, items, config)
    {
        var hasWorkflowItems = false;
        for (var i in config.workflowActions)
        {
            var $btn = AmetysFrontEditionContentToolbar._getWorkflowActionButton(i, config, {});
            if ($btn)
            {
                $btn.appendTo(toolbar);
            }
        }
    },
    
    /**
     * @private
     * Get a button for a workflow action
     * @param {Number} id the workflow action id
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button options
     * @return The button or null if user has no right to execute this action
     */
    _getWorkflowActionButton: function (id, config, opts)
    {
        if (config.workflowActions[id])
        {
            if ($j.inArray(id, config.excludedWorkflowActionsOutsideEditionMode) != -1 && !AmetysFrontEdition.EDITION_MODE)
            {
                // action is not autorized if not in edition mode
                return null;
            }
            
            return AmetysFrontEditionContentToolbar._getActionToolbarButton({
                iconCls: opts.hasOwnProperty('icon') ? opts.icon : config.workflowActions[id].icon,
                label: opts.hasOwnProperty('label') ? opts.label : config.workflowActions[id].label,
                handler: function() {
	                AmetysFrontEdition.execWorkflowAction(config.contentId, id);
	            },
                type: 'workflow'
            });
        }
        return null;
    },
    
    /**
     * @private
     * Add available workflow items to menu
     * @param {Object} items The menu items
     * @param {Object} config The toolbar configuration
     * @return {Boolean} true if at least one item was added
     */
    _addWorkflowActionsItems: function (items, config)
    {
        var hasWorkflowItems = false;
        for (var i in config.workflowActions)
        {
            hasWorkflowItems = AmetysFrontEditionContentToolbar._addWorkflowActionItem(i, items, config, {}) || hasWorkflowItems;
        }
        return hasWorkflowItems;
    },
    
    /**
     * @private
     * Add a workflow item to menu
     * @param {Number} actionId The workflow action id to add
     * @param {Object} items The menu items
     * @param {Object} config The toolbar configuration
     * @param {Object} opts The item options
     * @return {Boolean} true if the item was added
     */
    _addWorkflowActionItem: function (actionId, items, config, opts)
    {
        if (config.workflowActions[actionId])
        {
            if ($j.inArray(actionId, config.excludedWorkflowActionsOutsideEditionMode) != -1 && !AmetysFrontEdition.EDITION_MODE)
            {
                // action is not autorized if not in edition mode
                return null;
            }
            
            items["workflow-action-" + actionId] = {
               name: opts.hasOwnProperty('label') ? opts.label : config.workflowActions[actionId].label,
               icon: opts.hasOwnProperty('icon') ? opts.icon : config.workflowActions[actionId].icon,
               callback: AmetysFrontEdition.execWorkflowAction.bind(AmetysFrontEdition, config.contentId, actionId, null)
            };  
            
            return true;
            
        }
        return false;
    },
    
    /**
     * @private
     * Get a button for a action
     * @param {Object} config the button configuration
     * @param {String} config.icon the css class for icon
     * @param {label} config.label the button's label
     * @param {Boolean} [config.hideLabel] true to hide label
     * @param {Function} [config.handler] the function to call on click
     * @param {String} [config.type] the type of button
     * @param {String} [config.id] the id of button
     * @param {Boolean} [config.openDialog] true if the handler of button will open a dialog
     * @return The button
     */
    _getActionToolbarButton: function(config)
    {
        return AmetysFrontEditionToolbarHelper.createToolbarButton($j.extend(config, {
            className: AmetysFrontEditionContentToolbar.ACTION_BUTTON_CLASSNAMES + (config.type ? ' btn-type-' + config.type : ''),
        }));
    },
    
    /**
     * @private
     * Get a button for a status
     * @param {Number} id The step id
     * @param {String} label The step label
     * @param {Object} stepsCfg the steps configuration
     * @return The button
     */
    _getStatusToolbarButton: function(id, label, stepsCfg)
    {
        var cls = AmetysFrontEditionContentToolbar.STATUS_BUTTON_CLASSNAMES;
        
        var type = stepsCfg[String(id)] ? stepsCfg[String(id)].type : 'neutral';
        var iconCls = stepsCfg[String(id)] ? stepsCfg[String(id)].icon : '';
        
        if (type == 'positive')
        {
            cls += ' ' + AmetysFrontEditionContentToolbar.STATUS_POS_BUTTON_CLASSNAME;
            iconCls = iconCls ? iconCls : "fas fa-check";
        }
        else if (type == 'negative')
        {
            cls += ' ' + AmetysFrontEditionContentToolbar.STATUS_NEG_BUTTON_CLASSNAME;
            iconCls = iconCls ? iconCls : "fas fa-times";
        }
        else
        {
            cls += ' ' + AmetysFrontEditionContentToolbar.STATUS_NEUTRAL_BUTTON_CLASSNAME;
        }
        
        return AmetysFrontEditionToolbarHelper.createToolbarButton({
            id : id,
            className: cls,
            iconCls: iconCls,
            label: label,
            disabled: true,
            pressed: true
        });
    },
    
    /**
     * @private
     * Add step buttons representing the history status
     * @param {Object} toolbar the toolbar
     * @param {Object} statusHistory the status history
     * @param {Object} stepsCfg the steps configuration
     */
    _addHistoryStatusToolbarButtons: function(toolbar, statusHistory, stepsCfg)
    {
        for (var stepId in statusHistory)
        {
            var $btn = AmetysFrontEditionContentToolbar._getStatusToolbarButton(stepId, statusHistory[stepId].label, stepsCfg);
            if ($btn)
            {
                $btn.appendTo(toolbar);
            }
        }
    },
    
    /**
     * @private
     * Get the button to edit the content
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getEditContentButton: function (config, opts)
    {
        var btnConfig = {
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-pen',
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_EDIT_CONTENT_BUTTON}}",
            type: 'edit',
            openDialog: true,
            handler: function() {
                AmetysFrontEdition.editContent(config.contentId, AmetysFrontEdition.editActionId, config.editViewName);
            }
        }
        return AmetysFrontEditionContentToolbar._getActionToolbarButton(btnConfig);
    },
    
    /**
     * @private
     * Return a menu item to edit a content
     * @param {Object} config the toolbar configuration
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getEditContentMenuItem: function (config, opts)
    {
        return {
            name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_EDIT_CONTENT_BUTTON}}", 
            icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-pen",
            callback: function(itemKey, opt, e) {
                AmetysFrontEdition.editContent(config.contentId, AmetysFrontEdition.editActionId, config.editViewName);
            }
        };
    },
    
    /**
     * @private
     * Get the button to tag the content
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getTagContentButton: function (config, opts)
    {
        var btnConfig = {
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-tag',
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TAG_CONTENT_BUTTON}}",
            type: 'tag',
            openDialog: true,
            handler: function() {
	            AmetysFrontEdition.page.changeContentTags(config.contentId)
	        }
        }
        
        return AmetysFrontEditionContentToolbar._getActionToolbarButton(btnConfig);
    },
    
    /**
     * @private
     * Menu item to tag a content
     * @param {Object} config 
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getTagContentMenuItem: function (config, opts)
    {   
        return {
            name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TAG_CONTENT_BUTTON}}", 
            icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-tag",
            callback: function(itemKey, opt, e) {
                AmetysFrontEdition.page.changeContentTags(config.contentId)
            }
        };
    },
    
    /**
     * @private
     * Get the button to remove the content
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getRemoveContentButton: function (config, opts)
    {
        var btnConfig = {
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-times',
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_REMOVE_CONTENT_BUTTON}}",
            type: 'remove',
            handler: function() {
	             AmetysFrontEdition.page.removeZoneItem({
	                zoneItemId: config.zoneItemId,
	                callback: opts.callback
	             });
	        }
        }
        return AmetysFrontEditionContentToolbar._getActionToolbarButton(btnConfig);
    },
    
    /**
     * @private
     * Get the button to delete the content
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getDeleteContentButton: function (config, opts)
    {
        var btnConfig = {
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-trash-alt',
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_DELETE_CONTENT_BUTTON}}",
            type: 'delete',
            handler: function() {
                AmetysFrontEdition.deleteContent(config.contentId, opts.callback);
            }
        }
        
        return AmetysFrontEditionContentToolbar._getActionToolbarButton(btnConfig);
    },
    
    /**
     * @private
     * Menu item to remove a content
     * @param {Object} config 
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getRemoveContentMenuItem: function(config, opts)
    {
        return {
            name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_REMOVE_CONTENT_BUTTON}}", 
            icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-times",
            callback: function(itemKey, opt, e) {
                AmetysFrontEdition.page.removeZoneItem({
                    zoneItemId: config.zoneItemId,
                    callback: opts.callback
                });
            }
        };
    },
    
    /**
     * @private
     * Menu item to delete a content
     * @param {Object} config 
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getDeleteContentMenuItem: function(config, opts)
    {
        return {
            name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_DELETE_CONTENT_BUTTON}}", 
            icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-trash-alt",
            callback: function(itemKey, opt, e) {
                AmetysFrontEdition.deleteContent(config.contentId, opts.callback);
            }
        };
    },
    
    /**
     * @private
     * Get the button to move the content, with submenu to choose direction (up or down)
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getMoveZoneItemButton: function (config, opts)
    {
        var btnConfig = {
            id: 'ametys-front-edition-content-toolbar-btn-move-' + Math.random().toString(16).slice(-8),
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-arrows-alt',
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_MOVE_MENU_LABEL}}",
            type: 'move'
        }
        
        var $btn = AmetysFrontEditionContentToolbar._getActionToolbarButton(btnConfig);
        
        var id = 'ametys-front-edition-content-toolbar-btn-move-' + Math.random().toString(16).slice(-8);
        $btn.attr('id', id);
        
        AmetysFrontEditionContentToolbar._createMenu('#' + id, AmetysFrontEditionContentToolbar._getMoveZoneItemMenuItem(config, opts).items, false);
        
        return $btn;
    },
    
    /**
     * @private
     * Get the menu item to move the content, with submenu to choose direction (up or down)
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getMoveZoneItemMenuItem: function(config, opts)
	{
	    return {
	            name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_MOVE_MENU_LABEL}}", 
	            icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-arrows-alt",
	            items: {
	                "up": {
	                    name: "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_MOVE_MENU_UP_LABEL}}", 
	                    icon: "fas fa-arrow-up",
	                    callback: function(itemKey, opt, e) {
	                        AmetysFrontEdition.page.moveZoneItem(config.zoneItemId, config.zoneName, -1);
	                    },
                        disabled: !config.actionStatus['move-up'],
                        className: !config.actionStatus['move-up'] ? 'disabled' : ''
	                },
	                "down": {
	                    name: "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_MOVE_MENU_DOWN_LABEL}}", 
	                    icon: "fas fa-arrow-down",
	                    callback: function(itemKey, opt, e) {
	                        AmetysFrontEdition.page.moveZoneItem(config.zoneItemId, config.zoneName, 1);
	                    },
                        disabled: !config.actionStatus['move-down'],
                        className: !config.actionStatus['move-down'] ? 'disabled' : ''
	                }
	            }
	    }
	},
    
    /**
     * @private
     * Get the button to move up the content
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getMoveUpZoneItemButton: function (config, opts)
    {
        var btnConfig = {
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-arrow-up',
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_MOVE_CONTENT_UP}}",
            type: 'move-up',
            handler: function() {
	             AmetysFrontEdition.page.moveZoneItem(config.zoneItemId, config.zoneName, -1);
	        } 
        }
        
        return AmetysFrontEditionContentToolbar._getActionToolbarButton(btnConfig);
    },
    
    /**
     * @private
     * Menu item to move up a content
     * @param {Object} config 
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getMoveUpZoneItemMenuItem: function(config, opts)
    {
        return {
                name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_MOVE_CONTENT_UP}}", 
                icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-arrow-up",
                callback: function(itemKey, opt, e) {
                    AmetysFrontEdition.page.moveZoneItem(config.zoneItemId, config.zoneName, -1);
                }
        }
    },
    
    /**
     * @private
     * Get the button to move down the content
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getMoveDownZoneItemButton: function (config, opts)
    {
        var btnConfig = {
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-arrow-down',
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_MOVE_CONTENT_DOWN}}",
            type: 'move-down',
            handler: function() {
                 AmetysFrontEdition.page.moveZoneItem(config.zoneItemId, config.zoneName, 1);
            } 
        }
        
        return AmetysFrontEditionContentToolbar._getActionToolbarButton(btnConfig);
    },
    
    /**
     * @private
     * Menu item to move down a content
     * @param {Object} config 
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getMoveDownZoneItemMenuItem: function(config, opts)
    {
        return {
                name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_MOVE_CONTENT_DOWN}}", 
                icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-arrow-down",
                callback: function(itemKey, opt, e) {
                    AmetysFrontEdition.page.moveZoneItem(config.zoneItemId, config.zoneName, 1);
                }
        }
    }
}

AmetysFrontEditionContentToolbar.CONTENT_ACTIONS_MENU_ITEM = {
        'edit': AmetysFrontEditionContentToolbar._getEditContentMenuItem,
        'tag': AmetysFrontEditionContentToolbar._getTagContentMenuItem,
        'move': AmetysFrontEditionContentToolbar._getMoveZoneItemMenuItem,
        'move-up': AmetysFrontEditionContentToolbar._getMoveUpZoneItemMenuItem,
        'move-down': AmetysFrontEditionContentToolbar._getMoveDownZoneItemMenuItem,
        'remove': AmetysFrontEditionContentToolbar._getRemoveContentMenuItem,
        'delete': AmetysFrontEditionContentToolbar._getDeleteContentMenuItem
}

AmetysFrontEditionContentToolbar.CONTENT_ACTIONS_BUTTON = {
        'edit': AmetysFrontEditionContentToolbar._getEditContentButton,
        'tag': AmetysFrontEditionContentToolbar._getTagContentButton,
        'move': AmetysFrontEditionContentToolbar._getMoveZoneItemButton,
        'move-up': AmetysFrontEditionContentToolbar._getMoveUpZoneItemButton,
        'move-down': AmetysFrontEditionContentToolbar._getMoveDownZoneItemButton,
        'remove': AmetysFrontEditionContentToolbar._getRemoveContentButton,
        'delete': AmetysFrontEditionContentToolbar._getDeleteContentButton
}