/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.frontnotification.userprefs;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.web.userpref.FOUserPreferencesConstants;

/**
 * Abstract classes for flash info user preferences
 *
 */
public abstract class AbstractItemsAction extends ServiceableAction
{
    /** The id of user preference for flash info contents */
    public static final String FLASHINFO_USER_PREF_ITEM_IDS = "flashInfoItemIds";
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The user preferences manager */
    protected UserPreferencesManager _userPrefManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _userPrefManager = (UserPreferencesManager) smanager.lookup(UserPreferencesManager.ROLE + ".FO");
    }

    /**
     * Get the ids of read flash infos 
     * @param user the user
     * @param siteName the site name
     * @return the ids of read contents
     * @throws UserPreferencesException if failed to retrieve user preferences
     */
    protected List<String> getReadItems(UserIdentity user, String siteName) throws UserPreferencesException
    {
        Map<String, String> contextVars = new HashMap<>();
        contextVars.put(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME, siteName);
        
        String contentIdsAsStr = _userPrefManager.getUserPreferenceAsString(user, "/sites/" + siteName, contextVars, FLASHINFO_USER_PREF_ITEM_IDS);
        if (StringUtils.isNotBlank(contentIdsAsStr))
        {
            return Arrays.asList(StringUtils.split(contentIdsAsStr, ","));
        }
        
        return Collections.emptyList();
    }
    
    /**
     * Save the ids of read items in user preferences
     * @param owner the user
     * @param itemIds the id of read items
     * @param siteName the site name
     * @throws UserPreferencesException if failed to save user preferences
     */
    protected void saveReadItems(UserIdentity owner, List<String> itemIds, String siteName) throws UserPreferencesException
    {
        Map<String, String> contextVars = new HashMap<>();
        contextVars.put(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME, siteName);
        
        _userPrefManager.addUserPreference(owner, "/sites/" + siteName, contextVars, FLASHINFO_USER_PREF_ITEM_IDS, StringUtils.join(itemIds, ","));
    }
    
}
