/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The glossary DAO.
 */
 Ext.define('Ametys.cms.glossary.GlossaryDAO', {
 	singleton: true,
 	
 	constructor: function(config)
 	{
		/**
		 * @callable
		 * @member Ametys.cms.glossary.GlossaryDAO
		 * @method createDefinition 
		 * This calls the method 'createDefinition' of the server DAO 'GlossaryDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.word The word
 		 * @param {String} parameters.variants The variants
 		 * @param {String} parameters.content The content
 		 * @param {String} parameters.displayOnText The value of displayOnText
 		 * @param {String} parameters.siteName The site name
 		 * @param {String} parameters.language The language
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.glossary.GlossaryDAO",
				methodName: "createDefinition",
	     		callback: {
	         		handler: this._createDefinitionCb
	     		},
				waitMessage: "{{i18n PLUGINS_GLOSSARY_DEFINITION_ADDING_WAIT}})",
				errorMessage: {
				    msg: "{{i18n PLUGINS_GLOSSARY_DEFINITION_ERROR_CREATE}}",
				    category: Ext.getClassName(this)
				}
			}
		);
		/**
		 * @callable
		 * @member Ametys.cms.glossary.GlossaryDAO
		 * @method editDefinition 
		 * This calls the method 'editDefinition' of the server DAO 'GlossaryDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the word to edit
 		 * @param {String} parameters.word The word
 		 * @param {String} parameters.variants The variants
 		 * @param {String} parameters.content The content
 		 * @param {String} parameters.displayOnText The value of displayOnText
 		 * @param {String} parameters.siteName The site name
 		 * @param {String} parameters.language The language
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
				role: "org.ametys.plugins.glossary.GlossaryDAO",
	     		methodName: "editDefinition",
	     		callback: {
	         		handler: this._editDefinitionCb
	     		},
	     		waitMessage: "{{i18n PLUGINS_GLOSSARY_DEFINITION_EDITING_WAIT}}",
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_GLOSSARY_DEFINITION_ERROR_EDIT}}",
	     			category: Ext.getClassName(this)
	     		}
			}
		);
		/**
		 * @callable
		 * @member Ametys.cms.glossary.GlossaryDAO
		 * @method deleteDefinitions 
		 * This calls the method 'deleteDefinitions' of the server DAO 'GlossaryDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String[]} parameters.ids The ids of definitions to delete
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.glossary.GlossaryDAO",
	     		methodName: "deleteDefinitions",
	     		callback: {
	         		handler: this._deleteDefinitionsCb
	     		},
	     		waitMessage: "{{i18n PLUGINS_GLOSSARY_DEFINITION_REMOVING_WAIT}}",
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_GLOSSARY_DEFINITION_ERROR_DELETE}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
 	},
 	
 	/**
 	 * Callback function called after {@link #createDefinition} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_createDefinitionCb: function(response)
 	{
		var error = response['error'];
		if (error == 'word-already-exists')
		{
			Ametys.Msg.show({
				title: "{{i18n PLUGINS_GLOSSARY_DEFINITION_ADD_TITLE}}",
				msg: "{{i18n PLUGINS_GLOSSARY_DEFINITION_ERROR_ALREADY_EXISTS}}",
				buttons: Ext.Msg.OK,
				icon: Ext.MessageBox.ERROR
			});
		}
		else
		{
	    	Ext.create("Ametys.message.Message", {
	    		type: Ametys.message.Message.CREATED,
	    		targets: {
	    			id: 'wordDefinition',
	    			parameters: {
	    				id: response.id
	    			}
	    		}
	    	});
		}
 	},
 	
 	/**
 	 * Callback function called after {@link #editDefinition} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_editDefinitionCb: function(response)
 	{
		var error = response['error'];
		if (error == 'word-already-exists')
        {
            Ametys.Msg.show({
                   title: "{{i18n PLUGINS_GLOSSARY_DEFINITION_EDIT_TITLE}}",
                   msg: "{{i18n PLUGINS_GLOSSARY_DEFINITION_ERROR_ALREADY_EXISTS}}",
                   buttons: Ext.Msg.OK,
                   icon: Ext.MessageBox.ERROR
            });
        }
        else if (error == 'unknown-definition')
        {
        	Ametys.Msg.show({
                title: "{{i18n PLUGINS_GLOSSARY_DEFINITION_EDIT_TITLE}}",
                msg: "{{i18n PLUGINS_GLOSSARY_DEFINITION_ERROR_UNKNOWN}}",
                buttons: Ext.Msg.OK,
                icon: Ext.MessageBox.ERROR
            });
        }
        else
        {
			Ext.create("Ametys.message.Message", {
				type: Ametys.message.Message.MODIFIED,
				targets: {
				    id: 'wordDefinition',
					parameters: {
						id: response.id
					}
				}
			});
        }
 	},
 	
 	/**
 	 * Callback function called after {@link #deleteDefinitions} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_deleteDefinitionsCb: function(response)
 	{
	    var error = response['error'];
	    if (error == 'unknown-definition')
	    {
	    	var unknownIds = response['unknown-ids']; 
	        Ametys.Msg.show({
	            title: "{{i18n PLUGINS_GLOSSARY_DEFINITION_DELETE_TITLE}}",
	            msg: "{{i18n PLUGINS_GLOSSARY_DEFINITION_ERROR_UNKNOWN}}" + ": " + unknownIds.toString(),
	            buttons: Ext.Msg.OK,
	            icon: Ext.MessageBox.ERROR
	        });
	    }
	    else
	    {
		    var targets = [];
		    var ids = response['ids'];
		    for (var i = 0; i < ids.length; i++)
		    {
		        targets.push({
		            id: 'wordDefinition', 
		        	parameters: {'id': ids[i]}
		        });
		    }
		    
		    if (targets.length > 0)
		    {
			    Ext.create("Ametys.message.Message", {
			    	type: Ametys.message.Message.DELETED,
			    	targets: targets
			    });
		    }
	    }
	 }
 	
 });