/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton class to handle actions on word definitions.
 */
 Ext.define('Ametys.plugins.glossary.GlossaryActions', {
 	singleton: true,
 	
 	/**
     * @private
     * @property __UITOOL_GLOSSARY The id of link directory UI tool
     */
    __UITOOL_GLOSSARY: 'uitool-glossary',
    
 	/**
 	 * Creates a new word definition.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	addDefinition: function(controller)
 	{
 		var tool = Ametys.tool.ToolsManager.getTool(this.__UITOOL_GLOSSARY);
 		var lang = tool != null ? tool.getCurrentLanguage() : Ametys.cms.language.LanguageDAO.getCurrentLanguage();
 		
 		Ametys.plugins.glossary.WordDefinitionDialog.open ({lang: lang, callback: Ext.bind(this._addDefinitionCb, this)});
 	},
 	
 	/**
     * Callback function after adding the new definition
     * Open the UI tool if it is not yet opened.
     * @param {String} id The id of created definition
     */
    _addDefinitionCb: function (id)
    {
		var tool = Ametys.tool.ToolsManager.getTool(this.__UITOOL_GLOSSARY);
    	if (tool == null)
    	{
    		Ametys.tool.ToolsManager.openTool(this.__UITOOL_GLOSSARY, {selectedDefinitionIds: [id]});
    	}
    },
 	
 	/**
 	 * Edits a given word definition.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	editDefinition: function (controller)
 	{
 		var targets = Ametys.message.MessageBus.getCurrentSelectionMessage().getTargets();
		var target = Ametys.message.MessageTargetHelper.findTarget(targets, function(target) { return target.getId() == Ametys.message.MessageTarget.WORD_DEFINITION; });
		
		if (target == null)
		{
			return;
		}
			
		var wordDefinitionId = target.getParameters().id;
 		var tool = Ametys.tool.ToolsManager.getTool(this.__UITOOL_GLOSSARY);
 		var lang = tool != null ? tool.getCurrentLanguage() : Ametys.cms.language.LanguageDAO.getCurrentLanguage();
 		
 		Ametys.plugins.glossary.WordDefinitionDialog.open ({
 			mode: 'edit', 
 			id: wordDefinitionId, 
 			lang: lang
 		});
 	},
 	
 	/**
 	 * Deletes a given word definition.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	deleteDefinition: function(controller)
 	{
 		Ametys.Msg.confirm("{{i18n PLUGINS_GLOSSARY_DEFINITION_DELETE_TITLE}}",
 				"{{i18n PLUGINS_GLOSSARY_DEFINITION_DELETE_CONFIRM}}",
 				this._doDelete,
 				this
 		);
 	},
 	 	
 	/**
 	 * @private
 	 * The action to perform when the user clicks on a button from the deleting message box.
 	 * @param {String} btn The pressed button. Can only be 'yes'/'no'
 	 */
 	_doDelete: function(btn)
 	{
	 	if (btn == 'yes')
	    {
	        var targets = Ametys.message.MessageBus.getCurrentSelectionMessage().getTargets();
	        var allTargets = Ametys.message.MessageTargetHelper.findTargets(targets, function(target) { return target.getId() == Ametys.message.MessageTarget.WORD_DEFINITION; });
	        
	        if (allTargets.length == 0)
	        {
	            return;
	        }
	        
	        var ids = [];
	        for (var i = 0; i < allTargets.length; i++)
	        {
	            ids.push(allTargets[i].getParameters().id);
	        } 
	        
	        Ametys.cms.glossary.GlossaryDAO.deleteDefinitions([ids], Ext.emptyFn, {});
	    }
 	}
 	
 });