/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This tool displays the JCR glossary-themes in a tree 
 * @private
 */
Ext.define('Ametys.plugins.glossary.theme.tool.ThemesTool', {
    extend : 'Ametys.plugins.web.tag.TagsTool',

    _tagTooltipTpl : Ext.create('Ext.XTemplate', [
                         '<tpl if="description && description != \'\'">',
                            '{description}<br/>',
                         '</tpl>'
    ]),
    
    getContextualParameters: function ()
    {
        return {
            siteName: Ametys.getAppParameter('siteName'),
            language: this._tree ? this._tree.queryById('themes-tool-languages').getValue() : ""
        };
    },
    
    _getTopToolbarItems: function()
    {
        var langCombo = Ametys.cms.language.LanguageDAO.createComboBox({
            itemId: 'themes-tool-languages',
            fieldLabel: "{{i18n PLUGINS_GLOSSARY_UITOOL_LANGUAGE}}",
            flex: 1,
            
            synchronize: true,
            listeners: {
                'change': Ext.bind(this._onChangeLang, this)
            }
        });
        
        var items = this.callParent();
        
        Ext.Array.insert(items, 0, [langCombo])
        return items;
    },
    
    _onChangeLang: function()
    {
        this.refresh();
    },
       
    getTagProviderId: function()
    {
    	return 'org.ametys.plugins.glossary.theme.JCRThemeProvider';
    },
    
    getTagProviderEPId: function()
    {
    	return 'org.ametys.plugins.glossary.theme.ThemeProviderExtensionPoint';
    },
    
    getTagDAO: function()
    {
    	return 'org.ametys.plugins.glossary.theme.JCRThemesDAO';
    },
    
    getTagModel: function()
    {
    	return 'Ametys.plugins.glossary.theme.ThemeNode';
    },
    
    getTagTargetMessage: function()
    {
    	return Ametys.message.MessageTarget.GLOSSARY_THEME_TAG;
    },
    
    getRootTagTargetMessage: function()
    {
    	return Ametys.message.MessageTarget.GLOSSARY_THEME_TAG_ROOT;
    },
    
    _rootNodeLabel : "{{i18n PLUGINS_GLOSSARY_THEMES_TAGPROVIDER_JCR_ROOT_LABEL}}",
    
    _createStore: function()
    {
       var store = this.callParent();
       var me = this;
       store.on('beforeload', this._onBeforeLoad, me);
       return store;
    },
    
    _onBeforeLoad: function(store, operation, eOpts)
    {
        // update contextual parameters
        operation.getProxy().setExtraParam("contextualParameters", this.getContextualParameters());
    },

	_getTooltip: function(node)
	{
		var title = node.get('name');
		if (node.get("title") != null)
		{
			title = Ext.String.escapeHtml(node.get('title')) + " (" + title + ")";
		}
		
		var text = this._tagTooltipTpl.applyTemplate ({
			description: Ext.String.escapeHtml(node.get('description')).replace(/\n/g, "<br/>")
		});
		
		return {
			title: title,
			glyphIcon: node.get('tooltipIconCls'),
			imageWidth: 48,
			imageHeight: 48,
			text: text,
			inribbon: false
		};
	},
    
	_updateAdditionalInfo : function (node, response)
    {
		//No additional info
    }
});

Ext.define("Ametys.message.ThemeMessageTarget",
    {
		override: "Ametys.message.MessageTarget",

	    statics: 
	    {
	        /**
	         * @member Ametys.message.MessageTarget
	         * @readonly
	         * @property {String} GLOSSARY_THEME_TAG The target type is a tag. The expected parameters are:
	         * @property {String} GLOSSARY_THEME_TAG.id The id of tag
	         * @property {String} GLOSSARY_THEME_TAG.parentId The id of the parent of the tag
	         * @property {String} GLOSSARY_THEME_TAG.name The name of tag
	         */
	         GLOSSARY_THEME_TAG: "glossary-theme",
	
	        /**
	         * @member Ametys.message.MessageTarget
	         * @readonly
	         * @property {String} GLOSSARY_THEME_TAG_ROOT The target type is a tag root. The expected parameters are:
	         * @property {String} GLOSSARY_THEME_TAG_ROOT.id The id of root tag
	         * @property {String} GLOSSARY_THEME_TAG_ROOT.name The name of root tag 
	         */
	         GLOSSARY_THEME_TAG_ROOT: "glossary-theme-root"
	    }
    }
);
