/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.glossary;

import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.commons.lang3.ArrayUtils;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.SimpleAmetysObject;
import org.ametys.web.repository.SiteAwareAmetysObject;
import org.ametys.web.repository.site.Site;

/**
 * Repository implementation of a glossary definition.
 */
public class DefaultDefinition extends SimpleAmetysObject<DefaultDefinitionFactory> implements Definition, SiteAwareAmetysObject
{
    
    /** Constant for word attribute. */
    public static final String ATTRIBUTE_WORD = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":word";
    /** Constant for variants attribute. */
    public static final String ATTRIBUTE_VARIANTS = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":variants";
    /** Constant for content attribute. */
    public static final String ATTRIBUTE_CONTENT = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":content";
    /** Constant for display attribute. */
    public static final String ATTRIBUTE_DISPLAY = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":display";
    /** Constant for themes attribute. */
    public static final String ATTRIBUTE_THEMES = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":themes";
    
    /**
     * Create a {@link DefaultDefinition}.
     * 
     * @param node the node backing this {@link AmetysObject}.
     * @param parentPath the parent path in the Ametys hierarchy.
     * @param factory the {@link DefaultDefinitionFactory} which creates the AmetysObject.
     */
    public DefaultDefinition(Node node, String parentPath, DefaultDefinitionFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    @Override
    public String getWord() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(ATTRIBUTE_WORD).getString();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the word property.", e);
        }
    }

    @Override
    public void setWord(String word) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(ATTRIBUTE_WORD, word);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the word property.", e);
        }
    }

    @Override
    public Set<String> getVariants() throws AmetysRepositoryException
    {
        try
        {
            Set<String> variants = new LinkedHashSet<>();
            
            Value[] values = getNode().getProperty(ATTRIBUTE_VARIANTS).getValues();
            for (Value value : values)
            {
                variants.add(value.getString());
            }
            
            return variants;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the variants property.", e);
        }

    }

    @Override
    public void setVariants(Collection<String> variants) throws AmetysRepositoryException
    {
        try
        {
            String[] values = variants.toArray(new String[variants.size()]);
            getNode().setProperty(ATTRIBUTE_VARIANTS, values);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the variants property.", e);
        }
    }

    @Override
    public String getContent() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(ATTRIBUTE_CONTENT).getString();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the content property.", e);
        }
    }

    @Override
    public void setContent(String content) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(ATTRIBUTE_CONTENT, content);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the content property.", e);
        }
    }

    @Override
    public boolean displayOnText() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(ATTRIBUTE_DISPLAY).getBoolean();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the display property.", e);
        }
    }

    @Override
    public void setDisplayOnText(boolean displayOnText) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(ATTRIBUTE_DISPLAY, displayOnText);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the display property.", e);
        }
    }
    
    @Override
    public Set<String> getAllForms() throws AmetysRepositoryException
    {
        Set<String> allForms = new HashSet<>();
        
        allForms.add(getWord());
        allForms.addAll(getVariants());
        
        return allForms;
    }
    
    @Override
    public Site getSite() throws AmetysRepositoryException
    {
        return getParent().getParent().getParent().getParent().getParent();
    }
    
    @Override
    public String getSiteName() throws AmetysRepositoryException
    {
        return getSite().getName();
    }
    
    /**
     * Get the definition language.
     * @return the definition language.
     */
    public String getLanguage()
    {
        return getParent().getParent().getName();
    }
    
    @Override
    public String[] getThemes() throws AmetysRepositoryException
    {
        return _getListFromAttributeName(ATTRIBUTE_THEMES);
    }

    @Override
    public void setThemes(String[] themes) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(ATTRIBUTE_THEMES, themes);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("An error occurred while trying to set the property '" + ATTRIBUTE_THEMES + "'.", e);
        }
    }

    @Override
    public void removeTheme(String themeId) throws AmetysRepositoryException
    {
        try
        {
            String[] themes = getThemes();
            String[] updatedThemes = ArrayUtils.removeElement(themes, themeId);
            getNode().setProperty(ATTRIBUTE_THEMES, updatedThemes);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error removing theme of id " + themeId, e);
        }
    }
    
    /**
     * Retrieves the list of values corresponding to the attribute name passed as
     * parameter
     * 
     * @param attributeName the name of the attribute to retrieve
     * @return the list corresponding to the attribute name
     */
    private String[] _getListFromAttributeName(String attributeName)
    {
        try
        {
            if (getNode().hasProperty(attributeName))
            {
                Value[] values = getNode().getProperty(attributeName).getValues();

                String[] list = new String[values.length];

                for (int i = 0; i < values.length; i++)
                {
                    list[i] = values[i].getString();
                }

                return list;
            }
            else
            {
                return new String[0];
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("An error occurred while trying to get the property '" + attributeName + "'.", e);
        }
    }
    
}
