/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.glossary;

import java.io.IOException;
import java.text.Normalizer;
import java.text.Normalizer.Form;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.plugins.glossary.theme.ThemeExpression;
import org.ametys.plugins.glossary.theme.ThemesDAO;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;

/**
 * Generate the list of word definitions as XML.
 */
public class DefinitionsGenerator extends ServiceableGenerator
{
    private ThemesDAO _themesDAO;
    private AmetysObjectResolver _ametysObjectResolver;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _themesDAO = (ThemesDAO) serviceManager.lookup(ThemesDAO.ROLE);
        _ametysObjectResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String siteName = parameters.getParameter("siteName", (String) request.getAttribute("site"));
        String language = parameters.getParameter("language", (String) request.getAttribute("sitemapLanguage"));
        String letter = parameters.getParameter("letter", request.getParameter("letter"));
        letter = StringUtils.isBlank(letter) ? "A" : letter.toUpperCase();
        Page page = (Page) request.getAttribute(Page.class.getName());
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(ZoneItem.class.getName());
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        String[] themes = serviceParameters.getValue("themes", false, new String[0]);
        
        contentHandler.startDocument();
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("siteName", siteName);
        attrs.addCDATAAttribute("language", language);
        attrs.addCDATAAttribute("letter", letter);
        if (page != null)
        {
            attrs.addCDATAAttribute("page", page.getId());
        }
        XMLUtils.startElement(contentHandler, "wordDefinitions", attrs);
        
        Expression themeExpression = _getThemeExpression(siteName, language, themes);
        
        String definitionQuery = GlossaryHelper.getDefinitionQuery(siteName, language, themeExpression);
        try (AmetysObjectIterable<DefaultDefinition> definitions = _ametysObjectResolver.query(definitionQuery);)
        {
            for (DefaultDefinition definition : definitions)
            {
                saxDefinition(definition);
            }
        }
        
        XMLUtils.endElement(contentHandler, "wordDefinitions");
        contentHandler.endDocument();
    }

    /**
     * Get the list of theme expressions to filter the definitions with
     * @param siteName the site's name
     * @param language the site's language
     * @param themes the array of theme Ids to create expressions
     * @return a list of ThemeExpression
     */
    protected Expression _getThemeExpression(String siteName, String language, String[] themes)
    {
        ThemeExpression[] themeExpressions = Arrays.stream(themes)
            .filter(StringUtils::isNotBlank)
            .filter(id -> themeExists(id, siteName, language))
            .map(id -> new ThemeExpression(id))
            .toArray(ThemeExpression[]::new);
        
        return themeExpressions.length == 0 ? null : new OrExpression(themeExpressions);
    }
    
    /**
     * Generate a word definition.
     * @param definition the definition to generate.
     * @throws SAXException If an error occurs while generating the XML.
     */
    protected void saxDefinition(DefaultDefinition definition) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", definition.getId());
        
        String word = definition.getWord();
        if (StringUtils.isNotEmpty(word))
        {
            String firstLetter = Normalizer.normalize(word.substring(0, 1), Form.NFD).replaceAll("\\p{InCombiningDiacriticalMarks}+", "");
            attrs.addCDATAAttribute("firstLetter", firstLetter.toUpperCase());
        }
        attrs.addCDATAAttribute("word", definition.getWord());
        attrs.addCDATAAttribute("variantsText", StringUtils.join(definition.getVariants(), ", "));
        attrs.addCDATAAttribute("displayOnText", Boolean.toString(definition.displayOnText()));
        
        XMLUtils.startElement(contentHandler, "wordDefinition", attrs);
        
        // Variants.
        XMLUtils.startElement(contentHandler, "variants");
        
        for (String variant : definition.getVariants())
        {
            AttributesImpl variantAttrs = new AttributesImpl();
            variantAttrs.addCDATAAttribute("word", variant);
            XMLUtils.createElement(contentHandler, "variant", variantAttrs);
        }
        
        XMLUtils.endElement(contentHandler, "variants");
        
        // Content
        XMLUtils.createElement(contentHandler, "content", definition.getContent());
        
        XMLUtils.endElement(contentHandler, "wordDefinition");
    }
    
    /**
     * Verify the existence of a theme
     * @param themeName the id of the theme to verify
     * @param siteName the site's name
     * @param language the site's language
     * @return true if the theme exists, false otherwise
     */
    public boolean themeExists(String themeName, String siteName, String language)
    {
        if (StringUtils.isBlank(themeName))
        {
            return false;
        }
        Map<String, Object> contextualParameters = new HashMap<>();
        contextualParameters.put("language", language);
        contextualParameters.put("siteName", siteName);
        List<String> checkTags = _themesDAO.checkTags(List.of(themeName), false, Collections.EMPTY_MAP, contextualParameters);
        return !checkTags.isEmpty();
    }
}
