/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.glossary;

import java.text.Normalizer;
import java.text.Normalizer.Form;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.glossary.theme.ThemesDAO;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.site.SiteManager;

/**
 * DAO for manipulating glossary.
 *
 */
public class GlossaryDAO extends AbstractLogEnabled implements Serviceable, Component
{
    /** The Avalon role */
    public static final String ROLE = GlossaryDAO.class.getName();
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The themes DAO */
    ThemesDAO _themeDAO;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _themeDAO = (ThemesDAO) manager.lookup(ThemesDAO.ROLE);
    }
    
    /**
     * Get the word definitions 
     * @param siteName The site name
     * @param lang The language
     * @return The word definitions
     */
    @Callable (rights = "Glossary_Rights_Definitions_Handle")
    public Map<String, Object> getDefinitions(String siteName, String lang)
    {
        Map<String, Object> result = new HashMap<>();
        
        List<Map<String, Object>> definitions = new ArrayList<>();
        
        TraversableAmetysObject definitionsNode = GlossaryHelper.getDefinitionsNode(_siteManager.getSite(siteName), lang);
        
        try (AmetysObjectIterable<DefaultDefinition> wordDefinitions = definitionsNode.getChildren())
        {
            for (DefaultDefinition wordDefinition : wordDefinitions)
            {
                definitions.add(getDefinition(wordDefinition));
            }
        }
        
        result.put("definitions", definitions);
        
        return result;
    }
    
    /**
     * Get properties of a word definition
     * @param id the id of definition
     * @return the properties
     */
    @Callable (rights = "Glossary_Rights_Definitions_Handle")
    public Map<String, Object> getDefinition (String id)
    {
        DefaultDefinition wordDefinition = _resolver.resolveById(id);
        
        return getDefinition(wordDefinition);
    }
    
    /**
     * Get the definition in JSON 
     * @param wordDefinition the word definition
     * @return the JSON for the word definition
     */
    public Map<String, Object> getDefinition (DefaultDefinition wordDefinition)
    {
        Map<String, Object> properties = new HashMap<>();
        
        properties.put("id", wordDefinition.getId());
        properties.put("lang", wordDefinition.getLanguage());
        properties.put("siteName", wordDefinition.getSiteName());
        properties.put("word", wordDefinition.getWord());
        String firstLetter = Normalizer.normalize(wordDefinition.getWord().substring(0, 1), Form.NFD).replaceAll("\\p{InCombiningDiacriticalMarks}+", "");
        properties.put("firstLetter", firstLetter);
        properties.put("variants", wordDefinition.getVariants());
        properties.put("content", wordDefinition.getContent());
        properties.put("displayOnText", wordDefinition.displayOnText());
        
        // Themes
        List<Map<String, Object>> themesList = new ArrayList<>();
        for (String themeId : wordDefinition.getThemes())
        {
            try
            {
                I18nizableText themeTitle = _themeDAO.getThemeTitle(themeId, wordDefinition.getSiteName(), wordDefinition.getLanguage());
                Map<String, Object> themeData = new HashMap<>();
                themeData.put("id", themeId);
                themeData.put("label", themeTitle);
                themesList.add(themeData);
            }
            catch (UnknownAmetysObjectException e)
            {
                // Theme does not exist anymore
            }
        }
        
        properties.put("themes", themesList);
        
        return properties;
    }
    
    /**
     * Creates a word definition.
     * @param word The word
     * @param variants The variants
     * @param content The content
     * @param themes the themes
     * @param display The value of displayOnText
     * @param siteName The site name
     * @param language The language
     * @return The id of the created word, or an error
     */
    @Callable (rights = "Glossary_Rights_Definitions_Handle")
    public Map<String, String> createDefinition (String word, String variants, String content,  List<String> themes, Boolean display, String siteName, String language)
    {
        Map<String, String> result = new HashMap<>();
        
        String capitalizedWord = StringUtils.capitalize(word);
        List<String> variantList = new ArrayList<>();
        if (StringUtils.isNotBlank(variants))
        {
            for (String variant : StringUtils.split(variants, ","))
            {
                String trimVariant = variant.trim();
                if (StringUtils.isNotEmpty(trimVariant))
                {
                    variantList.add(StringUtils.capitalize(trimVariant));
                }
            }
        }
        
        // Check that the word doesn't already exist.
        if (_wordExists(capitalizedWord, siteName, language))
        {
            result.put("error", "word-already-exists");
            return result;
        }
        
        // Check that none of the variant already exists.
        for (String variant : variantList)
        {
            if (_wordExists(variant, siteName, language))
            {
                result.put("error", "word-already-exists");
                return result;
            }
        }
        
        ModifiableTraversableAmetysObject rootNode = GlossaryHelper.getDefinitionsNode(_siteManager.getSite(siteName), language);
        
        String originalName = NameHelper.filterName(capitalizedWord);
        
        // Find unique name
        String name = originalName;
        int index = 2;
        while (rootNode.hasChild(name))
        {
            name = originalName + "-" + (index++);
        }
        
        DefaultDefinition def = rootNode.createChild(name, DefaultDefinitionFactory.DEFINITION_NODE_TYPE);
        def.setWord(capitalizedWord);
        def.setVariants(variantList);
        def.setContent(content);
        def.setThemes(themes.toArray(new String[themes.size()]));
        def.setDisplayOnText(display);
        
        rootNode.saveChanges();
        
        result.put("id", def.getId());
        
        return result;
    }
    
    /**
     * Edits a word definition.
     * @param id The id of the word to edit
     * @param word The new word
     * @param variants The new variants
     * @param content The new content
     * @param themes the themes
     * @param display The new value of displayOnText
     * @param siteName The site name
     * @param language The language
     * @return The id of the edited word, or an error
     */
    @Callable (rights = "Glossary_Rights_Definitions_Handle")
    public Map<String, String> editDefinition (String id, String word, String variants, String content, List<String> themes, Boolean display, String siteName, String language)
    {
        Map<String, String> result = new HashMap<>();
        
        String capitalizedWord = StringUtils.capitalize(word);
        List<String> variantList = new ArrayList<>();
        if (StringUtils.isNotBlank(variants))
        {
            for (String variant : StringUtils.split(variants, ","))
            {
                String trimVariant = variant.trim();
                if (StringUtils.isNotEmpty(trimVariant))
                {
                    variantList.add(StringUtils.capitalize(variant.trim()));
                }
            }
        }
        
        try
        {
            DefaultDefinition definition = _resolver.resolveById(id);
            
            Set<String> allForms = definition.getAllForms();
            
            // If the word was changed, check that the new word doesn't already exist.
            if (!allForms.contains(word) && _wordExists(word, siteName, language))
            {
                return Collections.singletonMap("error", "word-already-exists");
            }
            
            // If the word was changed, check that none of the variant already exists.
            for (String variant : variantList)
            {
                if (!allForms.contains(variant) && _wordExists(variant, siteName, language))
                {
                    return Collections.singletonMap("error", "word-already-exists");
                }
            }
            
            definition.setWord(capitalizedWord.trim());
            definition.setVariants(variantList);
            definition.setContent(content);
            definition.setThemes(themes.toArray(new String[themes.size()]));
            definition.setDisplayOnText(display);
            
            definition.saveChanges();
            
            result.put("id", definition.getId());
        }
        catch (UnknownAmetysObjectException e)
        {
            result.put("error", "unknown-definition");
            getLogger().error("Unable to edit definition of id '" + id + ", because it does not not exist.", e);
        }
        
        return result;
    }
    
    /**
     * Deletes word definitions.
     * @param ids The ids of definitions to delete
     * @return The ids of the deleted words, the ids of the unknown words, and maybe an error
     */
    @Callable (rights = "Glossary_Rights_Definitions_Handle")
    public Map<String, Object> deleteDefinitions (ArrayList<String> ids)
    {
        Map<String, Object> result = new HashMap<>();
        List<String> deletedDefinitions = new ArrayList<>();
        List<String> unknownIds = new ArrayList<>();
        
        for (String id : ids)
        {
            try
            {
                DefaultDefinition definition = _resolver.resolveById(id);
                
                ModifiableAmetysObject parent = definition.getParent();
                definition.remove();
                
                parent.saveChanges();
                
                deletedDefinitions.add(id);
            }
            catch (UnknownAmetysObjectException e)
            {
                result.put("error", "unknown-definition");
                getLogger().error("Unable to delete the definition of id '" + id + ", because it does not exist.", e);
                unknownIds.add(id);
            }
        }
        
        result.put("ids", deletedDefinitions);
        result.put("unknown-ids", unknownIds);
        
        return result;
    }
    
    /**
     * Tests if the specified word exists in the glossary.
     * @param word The word to test.
     * @param siteName The site name.
     * @param language The language.
     * @return true if the word exists.
     * @throws AmetysRepositoryException if a repository error occurs.
     */
    protected boolean _wordExists(String word, String siteName, String language) throws AmetysRepositoryException
    {
        String xpathQuery = GlossaryHelper.getWordExistsQuery(siteName, language, word);
        
        try (AmetysObjectIterable<DefaultDefinition> defs = _resolver.query(xpathQuery))
        {
            return defs.iterator().hasNext();
        }
    }
}
