/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.glossary;

import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.web.repository.site.Site;

/**
 * Glossary helper.
 */
public final class GlossaryHelper
{
    
    /** The glossary page tag. */
    public static final String GLOSSARY_PAGE_TAG = "GLOSSARY";
    
    private static final String __PLUGIN_NODE_NAME = "glossary";
    
    private static final String __DEFINITIONS_NODE_NAME = "ametys:wordDefinitions";
    
    private GlossaryHelper()
    {
        // Hides the default constructor.
    }
    
    /**
     * Get the root plugin storage object.
     * @param site the site.
     * @return the root plugin storage object.
     * @throws AmetysRepositoryException if a repository error occurs.
     */
    public static ModifiableTraversableAmetysObject getPluginNode(Site site) throws AmetysRepositoryException
    {
        try
        {
            ModifiableTraversableAmetysObject pluginsNode = site.getRootPlugins();
            
            return getOrCreateNode(pluginsNode, __PLUGIN_NODE_NAME, "ametys:unstructured");
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the glossary plugin node for site " + site.getName(), e);
        }
    }
    
    /**
     * Get the word definitions storage node.
     * @param site the site
     * @param language the language.
     * @return the word definitions storage node.
     * @throws AmetysRepositoryException if a repository error occurs.
     */
    public static ModifiableTraversableAmetysObject getDefinitionsNode(Site site, String language) throws AmetysRepositoryException
    {
        try
        {
            // Get the root plugin node.
            ModifiableTraversableAmetysObject pluginNode = getPluginNode(site);
            
            // Get or create the language node.
            ModifiableTraversableAmetysObject langNode = getOrCreateNode(pluginNode, language, "ametys:unstructured");
            
            // Get or create the definitions container node in the language node and return it.
            return getOrCreateNode(langNode, __DEFINITIONS_NODE_NAME, DefaultDefinitionFactory.DEFINITION_ROOT_NODE_TYPE);
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the word definitions node for site " + site.getName() + " and language " + language, e);
        }
    }

    /**
     * Get the word definitions storage node.
     * @param siteName the site name.
     * @return the word definitions storage node.
     */
    public static String getPluginNodePath(String siteName)
    {
        return String.format("//element(%s, ametys:site)/ametys-internal:plugins/%s", siteName, __PLUGIN_NODE_NAME);
    }
    
    /**
     * Get the word definitions storage node.
     * @param siteName the site name.
     * @param language the language.
     * @return the word definitions storage node.
     */
    public static String getDefinitionsNodePath(String siteName, String language)
    {
        return getPluginNodePath(siteName) + "/"  + language + "/" + __DEFINITIONS_NODE_NAME;
    }
    
    /**
     * Creates the XPath query for glossary words corresponding to specified {@link Expression} and {@link SortCriteria}.
     * @param siteName the site name.
     * @param language the language
     * @param expression the query predicates. Can be null.
     * @return the created XPath query
     */
    public static String getDefinitionQuery(String siteName, String language, Expression expression)
    {
        SortCriteria sortCriteria = new SortCriteria();
        sortCriteria.addJCRPropertyCriterion(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":word", true, true);
        
        return getDefinitionQuery(siteName, language, expression, sortCriteria);
    }
    
    /**
     * Creates the XPath query for glossary words corresponding to specified {@link Expression} and {@link SortCriteria}.
     * @param siteName the site name.
     * @param language the language
     * @param expression the query predicates. Can be null.
     * @param sortCriteria criteria for sorting results. Can be null.
     * @return the created XPath query
     */
    public static String getDefinitionQuery(String siteName, String language, Expression expression, SortCriteria sortCriteria)
    {
        String predicats = null;
        
        if (expression != null)
        {
            predicats = StringUtils.trimToNull(expression.build());
        }
        
        String xpathQuery = getDefinitionsNodePath(siteName, language) + "/element(*, " + DefaultDefinitionFactory.DEFINITION_NODE_TYPE + ")" + (predicats != null ? "[" + predicats + "]" : "") + ((sortCriteria != null) ? (" " + sortCriteria.build()) : "");
        return xpathQuery;
    }
    
    /**
     * Get the word definitions storage node.
     * @param siteName the site name.
     * @param language the language.
     * @param word the searched word
     * @return the word definitions storage node.
     */
    public static String getWordExistsQuery(String siteName, String language, String word)
    {
        String lowerWord = StringUtils.replace(word, "'", "''").toLowerCase();
        return getDefinitionsNodePath(siteName, language) + "/element(*, " + DefaultDefinitionFactory.DEFINITION_NODE_TYPE
                + ")[fn:lower-case(@ametys-internal:word) = '" + lowerWord + "' or fn:lower-case(@ametys-internal:variants) = '" + lowerWord + "']";
    }
    
    /**
     * Get the word definitions storage node.
     * @param siteName the site name.
     * @param language the language.
     * @param letter the first letter.
     * @return the word definitions storage node.
     */
    public static String getFirstLetterDefQuery(String siteName, String language, String letter)
    {
        String lowerLetter = letter.toLowerCase();
        return getDefinitionsNodePath(siteName, language) + "/element(*, " + DefaultDefinitionFactory.DEFINITION_NODE_TYPE
                + ")[jcr:like(fn:lower-case(@ametys-internal:word), '" + lowerLetter + "%')] order by @ametys-internal:word";
    }
    
    /**
     * Get the word definitions storage node when first letter is not an alpha.
     * @param siteName the site name.
     * @param language the language.
     * @return the word definitions storage node.
     */
    public static String getNonAlphaFirstLetterDefQuery(String siteName, String language)
    {
        return getDefinitionsNodePath(siteName, language) + "/element(*, " + DefaultDefinitionFactory.DEFINITION_NODE_TYPE
                + ")[not(jcr:like(fn:lower-case(@ametys-internal:word), 'a%')) " 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'b%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'c%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'd%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'e%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'f%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'g%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'h%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'u%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'j%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'k%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'l%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'm%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'n%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'o%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'p%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'q%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'r%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 's%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 't%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'u%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'v%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'w%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'x%'))" 
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'y%'))"
                + "and not(jcr:like(fn:lower-case(@ametys-internal:word), 'z%'))"
                + "] order by @ametys-internal:word";
    }
    
    private static ModifiableTraversableAmetysObject getOrCreateNode(ModifiableTraversableAmetysObject parentNode, String nodeName, String nodeType) throws AmetysRepositoryException
    {
        ModifiableTraversableAmetysObject definitionsNode;
        if (parentNode.hasChild(nodeName))
        {
            definitionsNode = parentNode.getChild(nodeName);
        }
        else
        {
            definitionsNode = parentNode.createChild(nodeName, nodeType);
            parentNode.saveChanges();
        }
        return definitionsNode;
    }
    
}
