/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.glossary.theme;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.SimpleAmetysObject;
import org.ametys.web.repository.SiteAwareAmetysObject;
import org.ametys.web.repository.site.Site;

/**
 * Repository implementation of a glossary theme.
 */
public class DefaultTheme extends SimpleAmetysObject<DefaultThemeFactory> implements Theme, SiteAwareAmetysObject
{
    
    /** Constant for label attribute. */
    public static final String ATTRIBUTE_LABEL = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":label";
    
    /**
     * Create a {@link DefaultTheme}.
     * 
     * @param node the node backing this {@link AmetysObject}.
     * @param parentPath the parent path in the Ametys hierarchy.
     * @param factory the {@link DefaultThemeFactory} which creates the AmetysObject.
     */
    public DefaultTheme(Node node, String parentPath, DefaultThemeFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    @Override
    public String getLabel() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(ATTRIBUTE_LABEL).getString();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the label attribute.", e);
        }
    }
    
    @Override
    public void setLabel(String label) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(ATTRIBUTE_LABEL, label);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the label attribute.", e);
        }
    }
    
    @Override
    public Site getSite() throws AmetysRepositoryException
    {
        return getParent().getParent().getParent().getParent().getParent();
    }
    
    @Override
    public String getSiteName() throws AmetysRepositoryException
    {
        return getSite().getName();
    }
    
    /**
     * Get the theme language.
     * @return the theme language.
     */
    public String getLanguage()
    {
        return getParent().getParent().getName();
    }
    
}
