/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.glossary.theme;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.tag.Tag;
import org.ametys.cms.tag.TagProvider;
import org.ametys.cms.tag.jcr.AbstractJCRTagsDAO;
import org.ametys.cms.tag.jcr.JCRTag;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

/**
 * Component for operations on JCR tags
 */
public class JCRThemesDAO extends AbstractJCRTagsDAO
{
    /** The Avalon role */
    public static final String ROLE = JCRThemesDAO.class.getName();
    
    /** The tag provider extension point */
    protected ThemeProviderExtensionPoint _tagProviderExtPt;
    
    /** The rights manager */
    protected RightManager _rightManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _tagProviderExtPt = (ThemeProviderExtensionPoint) serviceManager.lookup(ThemeProviderExtensionPoint.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }
    
    @Override
    public ModifiableTraversableAmetysObject _getTagRootObject (String tagProviderId, Map<String, Object> contextualParameters) throws RepositoryException
    {
        JCRThemeProvider provider = (JCRThemeProvider) _tagProviderExtPt.getExtension(tagProviderId);
        return provider.getRootNode(contextualParameters);
    }
    
    @Override
    protected void _checkUserRight() throws IllegalStateException
    {
        String rightId = "Glossary_Rights_Themes_Handle";
        String context = "/cms";
        
        UserIdentity user = _currentUserProvider.getUser();
        if (_rightManager.hasRight(user, rightId, context) != RightResult.RIGHT_ALLOW)
        {
            getLogger().error("User '" + user + "' tried to access a privileged feature without convenient right. Should have right '" + rightId + "' on context '" + context + "'");
            throw new IllegalStateException("You have no right to access this feature.");
        }
    }

    @Override
    protected Tag _getTagFromName(String name, Map<String, Object> contextualParameters)
    {
        return _tagProviderExtPt.getTag(name, contextualParameters);
    }
    
    @Override
    protected JCRTag _createJCRTag(String parentId, String name, String title, String description, Map<String, Object> otherParameters, Map<String, Object> contextualParameters) throws RepositoryException
    {
        ModifiableTraversableAmetysObject parent = null;
        if (StringUtils.isEmpty(parentId))
        {
            parent = _getTagRootObject(JCRThemeProvider.class.getName(), contextualParameters);
        }
        else
        {
            parent = _resolver.resolveById(parentId);
        }
        
        if (parent.hasChild(name))
        {
            throw new RepositoryException("This theme already exists.");
        }
        ThemeJCR jcrTag = parent.createChild(name, ThemeFactory.TAG_NODETYPE);
        jcrTag.setTitle(title);
        jcrTag.setDescription(description);
        
        parent.saveChanges();
        
        return jcrTag;
    }
    
    @Override
    protected JCRTag _updateJCRTag(String tagId, String title, String description, Map<String, Object> otherParameters) throws UnknownAmetysObjectException
    {
        ThemeJCR jcrTag = _resolver.resolveById(tagId);
        jcrTag.setTitle(title);
        jcrTag.setDescription(description);
        
        jcrTag.saveChanges();
        
        return jcrTag;
    }

    @Override
    protected Set<TagProvider< ? extends Tag>> _getTagProviders()
    {
        Set<TagProvider<? extends Tag>> providers = new HashSet<>();
        
        Set<String> ids = _tagProviderExtPt.getExtensionsIds();
        for (String id : ids)
        {
            TagProvider<? extends Tag> provider = _tagProviderExtPt.getExtension(id);
            providers.add(provider);
        }
        
        return providers;
    }
    
    @Override
    // Change right required to call the callable
    @Callable(rights = "Glossary_Rights_Themes_Handle")
    public Map<String, Object> getTagRootNode(String tagProviderId, Map<String, Object> contextualParameters) throws ProcessingException
    {
        return super.getTagRootNode(tagProviderId, contextualParameters);
    }
    
    @Override
    // Change right required to call the callable
    @Callable(rights = "Glossary_Rights_Themes_Handle")
    public Map<String, Object> getTag(String tagId)
    {
        return super.getTag(tagId);
    }
}
