/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.glossary.theme;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.tag.AbstractTagsDAO;
import org.ametys.cms.tag.Tag;
import org.ametys.cms.tag.TagProvider;
import org.ametys.cms.tag.jcr.AbstractJCRTagsDAO;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * DAO for manipulating themes
 */
public class ThemesDAO extends AbstractTagsDAO
{
    /** The avalon role */
    public static final String ROLE = ThemesDAO.class.getName();

    /** The JCR Themes DAO */
    protected JCRThemesDAO _jcrThemesDAO;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _jcrThemesDAO = (JCRThemesDAO) manager.lookup(JCRThemesDAO.ROLE);
    }
    
    @Override
    public String getTagProviderEPRole()
    {
        return ThemeProviderExtensionPoint.ROLE;
    }
    
    @Override
    protected AbstractJCRTagsDAO _getTagJCRDAO()
    {
        return _jcrThemesDAO;
    }

    @Override
    protected List<TagProvider< ? extends Tag>> getCustomTagProvider()
    {
        List<TagProvider<? extends Tag>> providers = new ArrayList<>();
        providers.add(_tagProviderExtPt.getExtension(JCRThemeProvider.class.getName()));
        
        return providers;
    }
    
    /**
     * Get theme's title from its name
     * @param themeName the theme name
     * @param siteName the site's name
     * @param language the site's language
     * @return the title of the theme. Null if the theme doesn't exist
     */
    public I18nizableText getThemeTitle(String themeName, String siteName, String language)
    {
        Map<String, Object> contextualParameters = new HashMap<>();
        contextualParameters.put("language", language);
        contextualParameters.put("siteName", siteName);
        if (themeExists(themeName, siteName, language))
        {
            Tag tag = getTag(themeName, contextualParameters);
            return tag.getTitle();
        }
        else
        {
            getLogger().warn("Can't find theme with name " + themeName + " for site " + siteName + " and language " + language);
        }
            
        return null;
    }
    
    /**
     * Verify the existence of a theme
     * @param themeName the id of the theme to verify
     * @param siteName the site's name
     * @param language the site's language
     * @return true if the theme exists, false otherwise
     */
    public boolean themeExists(String themeName, String siteName, String language)
    {
        if (StringUtils.isBlank(themeName))
        {
            return false;
        }
        Map<String, Object> contextualParameters = new HashMap<>();
        contextualParameters.put("language", language);
        contextualParameters.put("siteName", siteName);
        List<String> checkTags = checkTags(List.of(themeName), false, Collections.EMPTY_MAP, contextualParameters);
        return !checkTags.isEmpty();
    }
}
