/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.glossary.transformation;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.transformation.I18nTransformer;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.cms.transformation.AbstractEnhancementHandler;
import org.ametys.cms.transformation.URIResolverExtensionPoint;
import org.ametys.plugins.glossary.DefaultDefinition;
import org.ametys.plugins.glossary.Definition;
import org.ametys.plugins.glossary.GlossaryHelper;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.web.WebHelper;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.tags.TagExpression;

/**
 * Definition enhancement handler.
 */
public class DefinitionEnhancementHandler extends AbstractEnhancementHandler implements Component, Serviceable, Contextualizable
{
    /** The Avalon role. */
    public static final String ROLE = DefinitionEnhancementHandler.class.getName();
    
    /** The tags in which to ignore the glossary words. */
    private static final Set<String> __IGNORE_TAGS = new HashSet<>();
    static
    {
        __IGNORE_TAGS.add("head");
        __IGNORE_TAGS.add("script");
        __IGNORE_TAGS.add("style");
        __IGNORE_TAGS.add("option");
        __IGNORE_TAGS.add("a");
        __IGNORE_TAGS.add("h1");
        __IGNORE_TAGS.add("h2");
        __IGNORE_TAGS.add("h3");
        __IGNORE_TAGS.add("h4");
        __IGNORE_TAGS.add("h5");
        __IGNORE_TAGS.add("h6");
    }
    
    /** The namespaces in which to ignore the glossary words. */
    private static final Set<String> __IGNORE_NAMESPACES = new HashSet<>();
    static
    {
        __IGNORE_NAMESPACES.add(I18nTransformer.I18N_NAMESPACE_URI);
    }
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The page URI resolver. */
    protected URIResolverExtensionPoint _uriResolver;
    
    /** The avalon context. */
    protected Context _context;
    
    /** The word definitions. */
    protected Map<String, Definition> _definitions;
    
    /** The glossary page href. */
    protected String _glossaryHref;
    
    /** Ignored namespace stack. */
    protected Map<String, Integer> _ignoredNamespaceStack;
    
    /** True if we are processing a paragraph, false otherwise. */
    private int _inIgnoredTag;
    
    private RenderingContextHandler _renderingContextHandler;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _uriResolver = (URIResolverExtensionPoint) serviceManager.lookup(URIResolverExtensionPoint.ROLE);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _renderingContextHandler = (RenderingContextHandler) serviceManager.lookup(RenderingContextHandler.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void startDocument() throws SAXException
    {
        super.startDocument();
        
        _glossaryHref = null;
        _inIgnoredTag = 0;
        
        _ignoredNamespaceStack = new HashMap<>();
        for (String ignoredNamespace : __IGNORE_NAMESPACES)
        {
            _ignoredNamespaceStack.put(ignoredNamespace, 0);
        }
    }
    
    @Override
    public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException
    {
        super.startElement(uri, localName, qName, atts);
        
        if (__IGNORE_TAGS.contains(localName.toLowerCase()))
        {
            _inIgnoredTag++;
        }
        if (__IGNORE_NAMESPACES.contains(uri))
        {
            _ignoredNamespaceStack.put(uri, _ignoredNamespaceStack.get(uri) + 1);
        }
    }
    
    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException
    {
        if (__IGNORE_NAMESPACES.contains(uri))
        {
            _ignoredNamespaceStack.put(uri, _ignoredNamespaceStack.get(uri) - 1);
        }
        if (__IGNORE_TAGS.contains(localName.toLowerCase()))
        {
            _inIgnoredTag--;
        }
        
        super.endElement(uri, localName, qName);
    }
    
    @Override
    public void characters(char[] ch, int start, int length) throws SAXException
    {
        if (_searchCharacters())
        {
            Request request = ContextHelper.getRequest(_context);
            Content content = (Content) request.getAttribute(Content.class.getName());
            _charactersWithDefinitions(ch, start, length, content);
        }
        else
        {
            super.characters(ch, start, length);
        }
    }
    
    /**
     * Test if the currently processed characters have to be searched for glossary words or ignored.
     * @return true to search the characters for glossary words, false to ignore.
     */
    protected boolean _searchCharacters()
    {
        // Test if we are in a tag to ignore.
        boolean search = _inIgnoredTag < 1 && !_inUnmodifiableContent;
        
        // Test if we are in a tag which has a namespace to ignore.
        Iterator<Integer> nsIt = _ignoredNamespaceStack.values().iterator();
        while (nsIt.hasNext() && search)
        {
            // If the "namespace level" is greater than 0, we are in an ignored namespace.
            if (nsIt.next() > 0)
            {
                search = false;
            }
        }
        
        return search;
    }
    
    /**
     * SAX characters, generating definition tags on words that are present in the glossary.
     * @param ch the characters from the XML document.
     * @param start the start position in the array.
     * @param length the number of characters to read from the array.
     * @param content the content.
     * @throws SAXException if an error occurs generating the XML.
     */
    protected void _charactersWithDefinitions(char[] ch, int start, int length, Content content) throws SAXException
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Get site name and language.
        String siteName = WebHelper.getSiteName(request, content);
        
        String language = content.getLanguage();
        if (language == null)
        {
            language = (String) request.getAttribute("sitemapLanguage");
        }
        
        if (language == null)
        {
            Map objectModel = ContextHelper.getObjectModel(_context);
            language = org.apache.cocoon.i18n.I18nUtils.findLocale(objectModel, "locale", null, Locale.getDefault(), true).getLanguage();
        }
        
        // Build a pattern to detect the glossary words.
        Map<String, Definition> words = _getDefinitions(siteName, language);
        if (!words.isEmpty())
        {
            Pattern pattern = _getWordsPattern(words.keySet());
            
            // Match the pattern.
            String str = new String(ch, start, length);
            Matcher matcher = pattern.matcher(str);
            
            int previousMatch = start;
            while (matcher.find())
            {
                // Get a link on the tagged glossary page.
                String pageLink = _getGlossaryPageHref(siteName, language);
                
                int startIndex = matcher.start();
                int endIndex = matcher.end();
                
                int wordIndex = start + startIndex;
                
                // The matched word.
                String word = str.substring(startIndex, endIndex).toLowerCase();
                Definition definition = words.get(word);
                
                // A null definition (word not present in the map) should never happen, but protect anyway.
                if (definition != null && StringUtils.isNotEmpty(word))
                {
                    String defContent = definition.getContent();
                    
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("title", defContent);
                    
                    // Generate all the characters until the matched word.
                    super.characters(ch, previousMatch, wordIndex - previousMatch);
                    
                    // Generate the definition tag.
                    XMLUtils.startElement(_contentHandler, "dfn", attrs);
                    
                    // If a glossary page is tagged in this site and sitemap, generate a link to it.
                    if (StringUtils.isNotEmpty(pageLink))
                    {
                        RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
                        if (!(currentContext == RenderingContext.BACK))
                        {
                            pageLink = pageLink + "?letter=" + word.charAt(0) + "#" + definition.getWord();
                        }
                        
                        AttributesImpl linkAttrs = new AttributesImpl();
                        linkAttrs.addCDATAAttribute("href", pageLink);
                        XMLUtils.startElement(_contentHandler, "a", linkAttrs);
                    }
                    
                    // Generate the word itself.
                    super.characters(ch, wordIndex, endIndex - startIndex);
                    
                    if (StringUtils.isNotEmpty(pageLink))
                    {
                        XMLUtils.endElement(_contentHandler, "a");
                    }
                    
                    XMLUtils.endElement(_contentHandler, "dfn");
                }
                
                previousMatch = start + endIndex;
            }
            
            // Generate the end of the input. This will generate the whole input, unchanged,
            // if no glossary word was present.
            super.characters(ch, previousMatch, start + length - previousMatch);
        }
        else
        {
            super.characters(ch, start, length);
        }
    }
    
    /**
     * Get all the words with definitions to display.
     * @param siteName the site name.
     * @param lang the language.
     * @return an exhaustive set of the words.
     */
    protected Map<String, Definition> _getDefinitions(String siteName, String lang)
    {
        if (_definitions == null)
        {
            _definitions = _getWordsAndDefinitions(siteName, lang);
        }
        
        return Collections.unmodifiableMap(_definitions);
    }
    
    /**
     * Get all the words with definitions to display.
     * @param siteName the site name.
     * @param lang the language.
     * @return an exhaustive set of the words.
     */
    protected Map<String, Definition> _getWordsAndDefinitions(String siteName, String lang)
    {
        Map<String, Definition> words = new HashMap<>();
        
        TraversableAmetysObject definitionsNode = GlossaryHelper.getDefinitionsNode(_siteManager.getSite(siteName), lang);
        AmetysObjectIterable<DefaultDefinition> definitions = definitionsNode.getChildren();
        
        for (DefaultDefinition definition : definitions)
        {
            if (definition.displayOnText())
            {
                for (String word : definition.getAllForms())
                {
                    words.put(word.toLowerCase(), definition);
                }
            }
        }
        
        return words;
    }
    
    /**
     * Get a regexp that matches any of the definition words.
     * @param words the words.
     * @return the pattern.
     */
    protected Pattern _getWordsPattern(Set<String> words)
    {
        StringBuilder pattern = new StringBuilder();
        pattern.append("\\b(?:");
        
        Iterator<String> wordIt = words.iterator();
        for (int i = 0; wordIt.hasNext(); i++)
        {
            if (i > 0)
            {
                pattern.append('|');
            }
            
            // Quote
            pattern.append("\\Q").append(wordIt.next()).append("\\E");
        }
        
        pattern.append(")\\b");
        
        return Pattern.compile(pattern.toString(), Pattern.CASE_INSENSITIVE);
    }
    
    /**
     * Get the glossary page in a given site and language.
     * @param siteName the site name.
     * @param language the language.
     * @return the glossary page.
     */
    protected String _getGlossaryPageHref(String siteName, String language)
    {
        if (_glossaryHref == null)
        {
            Page glossaryPage = _getGlossaryPage(siteName, language);
            if (glossaryPage != null)
            {
                // FIXME CMS-2611 Force absolute 
                Request request = ContextHelper.getRequest(_context);
                boolean absolute = request.getAttribute("forceAbsoluteUrl") != null ? (Boolean) request.getAttribute("forceAbsoluteUrl") : false;
                
                _glossaryHref = _uriResolver.getResolverForType("page").resolve(glossaryPage.getId(), false, absolute, false);
            }
            else
            {
                _glossaryHref = "";
            }
        }
        
        return _glossaryHref;
    }
    
    /**
     * Get the glossary page in a given site and language.
     * @param siteName the site name.
     * @param language the language.
     * @return the glossary page.
     */
    protected Page _getGlossaryPage(String siteName, String language)
    {
        Page page = null;
        
        Expression glossaryExpr = new TagExpression(Operator.EQ, GlossaryHelper.GLOSSARY_PAGE_TAG);
        String xpath = PageQueryHelper.getPageXPathQuery(siteName, language, null, glossaryExpr, null);
        
        try (AmetysObjectIterable<Page> pages = _resolver.query(xpath);)
        {
            Iterator<Page> it = pages.iterator();
            if (it.hasNext())
            {
                page = it.next();
                
                if (it.hasNext())
                {
                    getLogger().warn(String.format("More than one page is tagged 'GLOSSARY' in site %s and sitemap %s, please tag a single page.", siteName, language));
                }
            }
        }
        
        return page;
    }    
}
