/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.glpi.dynamic;

import java.util.Collections;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.glpi.GlpiTicket;
import org.ametys.plugins.glpi.TicketGlpiManager;
import org.ametys.plugins.linkdirectory.dynamic.AbstractInternalDynamicInformationGenerator;
import org.ametys.plugins.linkdirectory.dynamic.DynamicInformationException;
import org.ametys.plugins.linkdirectory.dynamic.DynamicInformationException.ExceptionType;
import org.ametys.runtime.i18n.I18nizableText;

/**
 *  Generates sax events for GLPI tickets information 
 */
public class TicketsDynamicInformationGenerator extends AbstractInternalDynamicInformationGenerator
{
    /** The ticket glpi component */
    private TicketGlpiManager _ticketGlpiManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _ticketGlpiManager = (TicketGlpiManager) serviceManager.lookup(TicketGlpiManager.ROLE);
    }

    @Override
    protected void saxShortValue() throws SAXException, DynamicInformationException
    {
        try
        {
            UserIdentity currentUser = getCurrentUser();
            
            if (currentUser == null)
            {
                throw new DynamicInformationException("Unable to retrieve open GLPI tickets, user is not connected", ExceptionType.UNAUTHORIZED);
            }
            else
            {
                int count = _ticketGlpiManager.getCountOpenTickets(currentUser);
                if (count != -1)
                {
                    XMLUtils.createElement(contentHandler, SHORT_VALUE, String.valueOf(count));
                }
                else
                {
                    throw new DynamicInformationException("Unable to retrieve open GLPI tickets", ExceptionType.UNKNOWN);
                }
            }
        }
        catch (SAXException | DynamicInformationException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new DynamicInformationException("Unable to sax short value for GLPI tickets", e);
        }
    }

    @Override
    protected void saxLongValue() throws SAXException, DynamicInformationException
    {
        try
        {
            UserIdentity currentUser = getCurrentUser();
            
            if (currentUser == null)
            {
                throw new DynamicInformationException("Unable to retrieve open GLPI tickets, user is not connected", ExceptionType.UNAUTHORIZED);
            }
            else
            {
                int openCount = _ticketGlpiManager.getCountOpenTickets(currentUser);
                if (openCount != -1)
                {
                    I18nizableText longValue = new I18nizableText("plugin.glpi", "PLUGINS_GLPI_DYNAMIC_INFORMATION_NB_TICKETS", Collections.singletonList(String.valueOf(openCount)));
                    longValue.toSAX(contentHandler, LONG_VALUE);
                }
                else
                {
                    throw new DynamicInformationException("Unable to retrieve open GLPI tickets", ExceptionType.UNKNOWN);
                }
            }
        }
        catch (SAXException | DynamicInformationException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new DynamicInformationException("Unable to retrieve the number of open GLPI tickets", e);
        }
    }

    @Override
    protected void saxTooltips() throws SAXException, DynamicInformationException
    {
        try
        {
            UserIdentity currentUser = getCurrentUser();
            
            if (currentUser == null)
            {
                throw new DynamicInformationException("Unable to retrieve open GLPI tickets, user is not connected", ExceptionType.UNAUTHORIZED);
            }
            else
            {
                List<GlpiTicket> openTickets = _ticketGlpiManager.getOpenTickets(currentUser);
                int maxItems = getMaxItems();
                
                for (int i = 0; (maxItems == -1 || i < maxItems) && i < openTickets.size(); i++)
                {
                    GlpiTicket glpiTicket = openTickets.get(i);
                    
                    int id = glpiTicket.getId();
                    String title = glpiTicket.getTitle();
                    int status = glpiTicket.getStatusId();
                    I18nizableText statusLabel = _ticketGlpiManager.getGlpiStatusLabel(status);
                    
                    XMLUtils.startElement(contentHandler, ITEM);
                    XMLUtils.createElement(contentHandler, ITEM_TITLE, String.valueOf(id));
                    XMLUtils.createElement(contentHandler, ITEM_SUMMARY, title != null ? title : "");
                    statusLabel.toSAX(contentHandler, ITEM_FOOTER);
                    XMLUtils.endElement(contentHandler, ITEM);
                }
            }
        }
        catch (SAXException | DynamicInformationException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new DynamicInformationException("Unable to retrieve open GLPI tickets information", e);
        }
    }
}
