/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.glpi;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.JSONUtils;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

/**
 * Connection and information from GLPI webservice
 */
public class TicketGlpiManager extends AbstractLogEnabled implements Component, Initializable, Serviceable, PluginAware
{
    /** Avalon ROLE. */
    public static final String ROLE = TicketGlpiManager.class.getName();

    private static final String __GLPI_INIT_SESSION = "/initSession/";

    private static final String __GLPI_SEARCH_USERS = "/search/User/";

    private static final String __GLPI_SEARCH_TICKET = "/search/Ticket/";

    private static final String __GLPI_KILL_SESSION = "/killSession/";

    /** Maximum cache size, in number of records. */
    protected long _maxCacheSize;

    /** The cache TTL in minutes. */
    protected long _cacheTtl;

    /**
     * The user information cache. The key of the cache is the user identity it self
     */
    protected LoadingCache<UserIdentity, Map<String, Object>> _cache;

    /**
     * The user information cache. The key of the cache is the user identity it self
     */
    protected Cache<String, Integer> _cacheIdentities;

    private JSONUtils _jsonUtils;

    private Map<Integer, I18nizableText> _glpiStatus;
    private Map<Integer, I18nizableText> _glpiType;

    private String _pluginName;

    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }

    @Override
    public void initialize()
    {
        // Tickets cache
        GlpiCacheLoader loader = new GlpiCacheLoader();

        Long maxCacheSizeConf = Config.getInstance().getValue("org.ametys.plugins.glpi.maxsize");
        Long maxCacheSize = (long) (maxCacheSizeConf != null ? maxCacheSizeConf.intValue() : 1000);

        Long cacheTtlConf = Config.getInstance().getValue("org.ametys.plugins.glpi.ttl");
        Long cacheTtl = (long) (cacheTtlConf != null && cacheTtlConf.intValue() >= 0 ? cacheTtlConf.intValue() : 60);

        CacheBuilder<Object, Object> cacheBuilder = CacheBuilder.newBuilder().expireAfterWrite(cacheTtl, TimeUnit.MINUTES);

        if (maxCacheSize > 0)
        {
            cacheBuilder.maximumSize(maxCacheSize);
        }

        _cache = cacheBuilder.build(loader);

        // Identities cache
        maxCacheSizeConf = Config.getInstance().getValue("org.ametys.plugins.glpi.maxsize.identities");
        maxCacheSize = (long) (maxCacheSizeConf != null ? maxCacheSizeConf.intValue() : 5000);

        cacheTtlConf = Config.getInstance().getValue("org.ametys.plugins.glpi.ttl.identities");
        cacheTtl = (long) (cacheTtlConf != null && cacheTtlConf.intValue() >= 0 ? cacheTtlConf.intValue() : 60);

        CacheBuilder<Object, Object> cacheIdentitiesBuilder = CacheBuilder.newBuilder().expireAfterWrite(cacheTtl, TimeUnit.MINUTES);

        if (maxCacheSize > 0)
        {
            cacheIdentitiesBuilder.maximumSize(maxCacheSize);
        }

        _cacheIdentities = cacheIdentitiesBuilder.build();
    }

    /**
     * Get the user collaboration information from the exchange server.
     * 
     * @param userIdentity the user identity.
     * @return the user collaboration information as a Map.
     * @throws Exception If an error occurred
     */
    protected Map<String, Object> loadUserInfo(UserIdentity userIdentity) throws Exception
    {
        Map<String, Object> userInfo = new HashMap<>();

        String glpiUrl = Config.getInstance().getValue("org.ametys.plugins.glpi.url");
        String usertoken = Config.getInstance().getValue("org.ametys.plugins.glpi.usertoken");
        String apptoken = Config.getInstance().getValue("org.ametys.plugins.glpi.apptoken");

        if (glpiUrl == null || usertoken == null || apptoken == null)
        {
            if (getLogger().isWarnEnabled())
            {
                getLogger().warn("Missing configuration: unable to contact the GLPI WebService Rest API, the configuration is incomplete.");
            }

            return userInfo;
        }

        RequestConfig requestConfig = RequestConfig.custom().setConnectTimeout(10000).setSocketTimeout(10000).build();

        try (CloseableHttpClient httpclient = HttpClientBuilder.create().setDefaultRequestConfig(requestConfig).useSystemProperties().build())
        {
            String sessionToken = _getGlpiSessionToken(httpclient, glpiUrl, usertoken, apptoken);
            if (sessionToken == null)
            {
                return userInfo;
            }

            try
            {
                Integer userId = _cacheIdentities.get(userIdentity.getLogin().toLowerCase(), new Callable<Integer>()
                {
                    @Override
                    public Integer call() throws Exception
                    {
                        return getUserIdentity(httpclient, userIdentity, glpiUrl, sessionToken, apptoken);
                    }
                });

                if (userId != null && userId != -1)
                {
                    Map<String, Object> glpiOpenTickets = getGlpiTickets(httpclient, glpiUrl, sessionToken, apptoken, userId);
                    if (glpiOpenTickets != null && !glpiOpenTickets.isEmpty())
                    {
                        userInfo.put("countOpenTickets", glpiOpenTickets.get("countOpenTickets"));
                        userInfo.put("openTickets", glpiOpenTickets.get("openTickets"));
                    }
                }
                else
                {
                    getLogger().debug("GPLI identity not found for user {}", userIdentity);
                }
            }
            finally
            {
                _killGlpiSessionToken(httpclient, glpiUrl, sessionToken, apptoken);
            }
        }

        return userInfo;
    }

    private String _getGlpiSessionToken(CloseableHttpClient httpclient, String glpiUrl, String usertoken, String apptoken) throws IOException, URISyntaxException
    {
        URIBuilder builder = new URIBuilder(glpiUrl + __GLPI_INIT_SESSION).addParameter("user_token", usertoken).addParameter("app_token", apptoken);
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Call GLPI webservice to init session : " + builder.build());
        }
        
        Map<String, Object> jsonObject = _callWebServiceApi(httpclient, builder.build(), true);
        if (jsonObject != null && jsonObject.containsKey("session_token"))
        {
            String token = (String) jsonObject.get("session_token");
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("GPLI WS returned session token '" + token + "'");
            }
            return token;
        }
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("GPLI WS returned no session token for user token '" + usertoken + "'");
        }

        return null;
    }

    /**
     * Get the user identity and fill the user identities cache
     * 
     * @param httpclient The http client to send a request to the webservice
     * @param userIdentity The current user identity
     * @param glpiUrl The GLPI Url
     * @param sessionToken The session token
     * @param apptoken The app token
     * @return The user identity corresponding, or null if not found
     * @throws Exception if an error occurred
     */
    @SuppressWarnings("unchecked")
    protected Integer getUserIdentity(CloseableHttpClient httpclient, UserIdentity userIdentity, String glpiUrl, String sessionToken, String apptoken) throws Exception
    {
        Long maxIdentitiesSize = Config.getInstance().getValue("org.ametys.plugins.glpi.maxsize.identities");
        URIBuilder builder = new URIBuilder(glpiUrl + __GLPI_SEARCH_USERS)
                .addParameter("range", "0-" + (maxIdentitiesSize != null ? maxIdentitiesSize : 1000))
                .addParameter("forcedisplay[0]", "1")
                .addParameter("forcedisplay[1]", "2")
                .addParameter("criteria[0][field]", "8")
                .addParameter("criteria[0][searchtype]", "contains")
                .addParameter("criteria[0][value]", "1")
                .addParameter("session_token", sessionToken)
                .addParameter("app_token", apptoken);
        Map<String, Object> jsonObject = _callWebServiceApi(httpclient, builder.build(), true);
        if (jsonObject != null && jsonObject.containsKey("data"))
        {
            List<Map<String, Object>> data = (List<Map<String, Object>>) jsonObject.get("data");
            Map<String, Integer> userIdentities = data.stream()
                    // Remove entries with illegal values
                    .filter(user -> StringUtils.isNotEmpty((String) user.get("1")) && user.containsKey("2"))
                    // Collect into a map
                    .collect(
                        Collectors.toConcurrentMap(
                            user -> ((String) user.get("1")).toLowerCase(),
                            user -> (Integer) user.get("2"),
                            // use a custom mapper to handle duplicate key as gracefully as possible
                            (int1, int2) -> {
                                getLogger().warn(String.format("GLPI id '%d' and '%d' are linked to the same login. '%d' will be ignored", int1, int2, int2));
                                return int1;
                            }));
            _cacheIdentities.putAll(userIdentities);
            
            if (userIdentities.containsKey(userIdentity.getLogin().toLowerCase()))
            {
                return userIdentities.get(userIdentity.getLogin().toLowerCase());
            }
        }
        return -1;
    }

    /**
     * Get the GLPI tickets
     * @param httpclient the HTTP client
     * @param glpiUrl The url of GLPI server
     * @param sessionToken The session token
     * @param apptoken The app token
     * @param userId The user id
     * @return The tickets
     * @throws IOException if an error occurred
     * @throws URISyntaxException if failed to build uri
     */
    protected Map<String, Object> getGlpiTickets(CloseableHttpClient httpclient, String glpiUrl, String sessionToken, String apptoken, Integer userId) throws IOException, URISyntaxException
    {
        StringBuilder uri = new StringBuilder(glpiUrl + __GLPI_SEARCH_TICKET + "?");
        
        uri.append(getTicketSearchQuery(userId))
            .append("&session_token=").append(sessionToken)
            .append("&app_token=").append(apptoken);
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Call GLPI webservice to search tickets : " + uri.toString());
        }
        
        Map<String, Object> jsonObject = _callWebServiceApi(httpclient, new URI(uri.toString()), true);
        if (jsonObject != null)
        {
            Map<String, Object> glpiTicketsInfo = new HashMap<>();
            if (jsonObject.containsKey("totalcount"))
            {
                glpiTicketsInfo.put("countOpenTickets", jsonObject.get("totalcount"));
            }
            if (jsonObject.containsKey("data"))
            {
                Object dataObject = jsonObject.get("data");
                List<GlpiTicket> glpiTickets = new ArrayList<>();
                if (dataObject instanceof List)
                {
                    @SuppressWarnings("unchecked")
                    List<Object> dataList = (List<Object>) dataObject;
                    for (Object object : dataList)
                    {
                        if (object instanceof Map)
                        {
                            @SuppressWarnings("unchecked")
                            Map<String, Object> ticketData = (Map<String, Object>) object;
                            glpiTickets.add(parseTicket(ticketData));
                        }
                    }
                    glpiTicketsInfo.put("openTickets", glpiTickets);
                }
            }
            return glpiTicketsInfo;
        }
        return null;
    }
    
    /**
     * Parse data into {@link GlpiTicket}
     * @param data the json data
     * @return the {@link GlpiTicket}
     */
    protected GlpiTicket parseTicket(Map<String, Object> data)
    {
        int ticketId = (int) data.get("2");
        String ticketTitle = (String) data.get("1");
        int status = (int) data.get("12");
        
        int type = data.containsKey("14") ? (int) data.get("14") : -1;
        String category = (String) data.get("7");
        
        GlpiTicket ticket = new GlpiTicket(ticketId, ticketTitle, status, type, category);
        return ticket;
    }
    
    /**
     * Get the part of rest API url for tickets search
     * @param userId The user id
     * @return The search query to concat to rest API url
     */
    protected String getTicketSearchQuery(Integer userId)
    {
        StringBuilder sb = new StringBuilder();
        
        // forcedisplay is used for data to return
        
        // To get all available search options, call GLPI_URL/listSearchOptions/Ticket?session_token=...
        
        sb.append("forcedisplay[0]=1") // title
            .append("&forcedisplay[1]=12") // status
            .append("&forcedisplay[2]=4") // user id
            .append("&forcedisplay[3]=2") // ticket id
            .append("&forcedisplay[4]=7") // category
            .append("&forcedisplay[6]=14") // type
            .append("&criteria[0][field]=12&criteria[0][searchtype]=0&criteria[0][value]=notold") // status=notold (unresolved)
            .append("&criteria[1][link]=AND")
            .append("&criteria[1][field]=4&criteria[1][searchtype]=equals&criteria[1][value]=" + userId) // current user is the creator
            .append("&sort=4"); // sort
        
        // Get tickets with unresolved status (notold)
        return sb.toString();
    }

    private void _killGlpiSessionToken(CloseableHttpClient httpclient, String glpiUrl, String sessionToken, String apptoken) throws IOException, URISyntaxException
    {
        URIBuilder builder = new URIBuilder(glpiUrl + __GLPI_KILL_SESSION).addParameter("session_token", sessionToken).addParameter("app_token", apptoken);
        _callWebServiceApi(httpclient, builder.build(), false);
    }

    private Map<String, Object> _callWebServiceApi(CloseableHttpClient httpclient, URI uri, boolean getJsonObject) throws IOException
    {
        HttpGet get = new HttpGet(uri);

        try (CloseableHttpResponse httpResponse = httpclient.execute(get))
        {
            if (!_isSuccess(httpResponse.getStatusLine().getStatusCode()))
            {
                String msg = null;
                if (httpResponse.getEntity() != null)
                {
                    ByteArrayOutputStream bos = new ByteArrayOutputStream();
                    try (InputStream is = httpResponse.getEntity().getContent())
                    {
                        IOUtils.copy(is, bos);
                    }
                    
                    msg = bos.toString("UTF-8");
                }
                
                getLogger().error("An error occurred while contacting the GLPI Rest API (status code : " + httpResponse.getStatusLine().getStatusCode() + "). Response is : " + msg);
                return null;
            }

            if (getJsonObject)
            {
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                try (InputStream is = httpResponse.getEntity().getContent())
                {
                    IOUtils.copy(is, bos);
                }

                if (getLogger().isDebugEnabled())
                {
                    getLogger().debug("GLPI webservice at uri " + uri + " returned : " + bos.toString("UTF-8"));
                }
                
                return _jsonUtils.convertJsonToMap(bos.toString("UTF-8"));
            }
        }

        return null;
    }

    private boolean _isSuccess(int statusCode)
    {
        return statusCode >= 200 && statusCode < 300;
    }

    /**
     * Get the number of open tickets
     * @param userIdentity The user identity
     * @return the number of unread mail
     */
    public int getCountOpenTickets(UserIdentity userIdentity)
    {
        if (userIdentity == null)
        {
            throw new IllegalArgumentException("User is not connected");
        }
        
        Map<String, Object> userInfo = _cache.getUnchecked(userIdentity);
        if (userInfo.get("countOpenTickets") != null)
        {
            return (Integer) userInfo.get("countOpenTickets");
        }
        else
        {
            return -1;
            //throw new IllegalArgumentException("Unable to get open tickets count. No user matches in GPLI with login " + userIdentity.getLogin());
        }
    }

    /**
     * Get all information about open tickets
     * @param userIdentity The user identity
     * @return List of all information about open tickets
     */
    @SuppressWarnings("unchecked")
    public List<GlpiTicket> getOpenTickets(UserIdentity userIdentity)
    {
        if (userIdentity == null)
        {
            throw new IllegalArgumentException("User is not connected");
        }
        
        Map<String, Object> userInfo = _cache.getUnchecked(userIdentity);
        if (userInfo.get("openTickets") != null)
        {
            return (List<GlpiTicket>) userInfo.get("openTickets");
        }
        else
        {
            return Collections.EMPTY_LIST;
            //throw new IllegalArgumentException("Unable to get open tickets. No user matches in GPLI with login " + userIdentity.getLogin());
        }
    }

    /**
     * Get the i18n label of a GLPI status
     * @param status The GLPI status
     * @return the label of status as {@link I18nizableText}
     */
    public I18nizableText getGlpiStatusLabel(int status)
    {
        if (_glpiStatus == null)
        {
            _glpiStatus = new HashMap<>();
            _glpiStatus.put(1, new I18nizableText("plugin." + _pluginName, "PLUGINS_GLPI_INCOMMING_STATUS"));
            _glpiStatus.put(2, new I18nizableText("plugin." + _pluginName, "PLUGINS_GLPI_ASSIGNED_STATUS"));
            _glpiStatus.put(3, new I18nizableText("plugin." + _pluginName, "PLUGINS_GLPI_PLANNED_STATUS"));
            _glpiStatus.put(4, new I18nizableText("plugin." + _pluginName, "PLUGINS_GLPI_WAITING_STATUS"));
            _glpiStatus.put(5, new I18nizableText("plugin." + _pluginName, "PLUGINS_GLPI_SOLVED_STATUS"));
            _glpiStatus.put(6, new I18nizableText("plugin." + _pluginName, "PLUGINS_GLPI_CLOSED_STATUS"));
            
        }
        return _glpiStatus.get(Integer.valueOf(status));
    }
    
    /**
     * Get the i18n label of a GLPI ticket type
     * @param type The GLPI type
     * @return the label of status as {@link I18nizableText}
     */
    public I18nizableText getGlpiTypeLabel(int type)
    {
        if (_glpiType == null)
        {
            _glpiType = new HashMap<>();
            _glpiType.put(1, new I18nizableText("plugin." + _pluginName, "PLUGINS_GLPI_INCIDENT_TYPE"));
            _glpiType.put(2, new I18nizableText("plugin." + _pluginName, "PLUGINS_GLPI_DEMAND_TYPE"));
            
        }
        return _glpiType.get(Integer.valueOf(type));
    }

    /**
     * The Glpi cache loader.
     */
    protected class GlpiCacheLoader extends CacheLoader<UserIdentity, Map<String, Object>>
    {
        @Override
        public Map<String, Object> load(UserIdentity userIdentity) throws Exception
        {
            return loadUserInfo(userIdentity);
        }
    }
}
