/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.hyperplanning.dynamic;

import java.time.format.FormatStyle;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.hyperplanning.HyperplanningManager;
import org.ametys.plugins.hyperplanning.HyperplanningManager.CancelledLesson;
import org.ametys.plugins.linkdirectory.dynamic.AbstractInternalDynamicInformationGenerator;
import org.ametys.plugins.linkdirectory.dynamic.DynamicInformationException;
import org.ametys.plugins.linkdirectory.dynamic.DynamicInformationException.ExceptionType;
import org.ametys.runtime.i18n.I18nizableDateTime;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Generate sax events for hyperplanning cancelled lessons informations
 */
public class CancelledLessonsDynamicInformationGenerator extends AbstractInternalDynamicInformationGenerator
{
    private HyperplanningManager _hyperplanningManager;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _hyperplanningManager = (HyperplanningManager) serviceManager.lookup(HyperplanningManager.ROLE);
    }
    
    @Override
    protected void saxShortValue() throws SAXException, DynamicInformationException
    {
        UserIdentity currentUser = getCurrentUser();
        
        if (currentUser == null)
        {
            throw new DynamicInformationException("Unable to retrieve hyperplanning data, user is not connected", ExceptionType.UNAUTHORIZED);
        }
        else
        {
            Set<CancelledLesson> cancelledLessons = _getLessons(currentUser);
            XMLUtils.createElement(contentHandler, SHORT_VALUE, String.valueOf(cancelledLessons.size()));
        }
    }

    @Override
    protected void saxLongValue() throws SAXException, DynamicInformationException
    {
        UserIdentity currentUser = getCurrentUser();
        
        if (currentUser == null)
        {
            throw new DynamicInformationException("Unable to retrieve hyperplanning data, user is not connected", ExceptionType.UNAUTHORIZED);
        }
        else
        {
            Set<CancelledLesson> cancelledLessons = _getLessons(currentUser);
            int count = cancelledLessons.size();
            if (count == 0)
            {
                I18nizableText longValue = new I18nizableText("plugin.hyperplanning", "PLUGIN_HYPERPLANNING_DYNAMIC_INFORMATION_CANCELLED_LESSONS_NO_ELEMENT");
                longValue.toSAX(contentHandler, LONG_VALUE);
            }
            else if (count == 1)
            {
                I18nizableText longValue = new I18nizableText("plugin.hyperplanning", "PLUGIN_HYPERPLANNING_DYNAMIC_INFORMATION_CANCELLED_LESSONS_NB_ELEMENT", Map.of("count", new I18nizableText(Integer.toString(count))));
                longValue.toSAX(contentHandler, LONG_VALUE);
            }
            else
            {
                I18nizableText longValue = new I18nizableText("plugin.hyperplanning", "PLUGIN_HYPERPLANNING_DYNAMIC_INFORMATION_CANCELLED_LESSONS_NB_ELEMENTS", Map.of("count", new I18nizableText(Integer.toString(count))));
                longValue.toSAX(contentHandler, LONG_VALUE);
            }
        }
    }

    @Override
    protected void saxTooltips() throws SAXException, DynamicInformationException
    {
        UserIdentity currentUser = getCurrentUser();
        
        if (currentUser == null)
        {
            throw new DynamicInformationException("Unable to retrieve hyperplanning data, user is not connected", ExceptionType.UNAUTHORIZED);
        }
        else
        {
            Set<CancelledLesson> cancelledLessons = _getLessons(currentUser);
            
            int maxItems = getMaxItems();
            cancelledLessons.stream()
                .limit(maxItems != -1 ? maxItems : cancelledLessons.size())
                .forEach(lesson -> {
                    try
                    {
                        XMLUtils.startElement(contentHandler, ITEM);
                    
                        XMLUtils.createElement(contentHandler, ITEM_TITLE, lesson.code() + " " + (StringUtils.isNotBlank(lesson.fullLabel()) ? lesson.fullLabel() : lesson.label()));
                        
                        I18nizableText summary = new I18nizableText("plugin.hyperplanning", "PLUGIN_HYPERPLANNING_DYNAMIC_INFORMATION_CANCELLED_LESSONS_SUMMARY", Map.of("date", new I18nizableDateTime(lesson.date(), lesson.date().getZone(), FormatStyle.MEDIUM)));
                        summary.toSAX(contentHandler, ITEM_SUMMARY);
                        XMLUtils.createElement(contentHandler, ITEM_FOOTER,  lesson.cancelRationale() + " " + lesson.cancelComment());
                        
                        XMLUtils.endElement(contentHandler, ITEM);
                    }
                    catch (SAXException e)
                    {
                        throw new RuntimeException(e);
                    }
                });
        }
    }
    
    private Set<CancelledLesson> _getLessons(UserIdentity currentUser)
    {
        try
        {
            return _hyperplanningManager.getUpcomingCancelledLessons(currentUser);
        }
        catch (Exception e)
        {
            throw new DynamicInformationException("An error occured while retrieving data from hyperplanning", e);
        }
    }
}
