/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.hyperplanning;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Predicate;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.HttpUtils;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.indexeducation.hyperplanning.ApiClient;
import com.indexeducation.hyperplanning.ApiException;
import com.indexeducation.hyperplanning.api.CoursAnnulesApi;
import com.indexeducation.hyperplanning.api.CoursApi;
import com.indexeducation.hyperplanning.api.EtudiantsApi;
import com.indexeducation.hyperplanning.api.IcalsApi;
import com.indexeducation.hyperplanning.api.MatieresApi;
import com.indexeducation.hyperplanning.model.Cours;
import com.indexeducation.hyperplanning.model.CoursAnnules;
import com.indexeducation.hyperplanning.model.CoursAnnulesCleDetailSeancesPlaceesGet200ResponseInner;
import com.indexeducation.hyperplanning.model.Etudiants;
import com.indexeducation.hyperplanning.model.Matieres;

/**
 * Component handling the communication with a remote hyperplanning server
 */
public class HyperplanningManager extends AbstractLogEnabled implements Initializable, Component, Serviceable
{
    /** The avalon role */
    public static final String ROLE = HyperplanningManager.class.getName();
    private static final String __CANCELLED_LESSONS_CACHE = HyperplanningManager.class.getName() + "$cancelledLessons";
    private static final String __STUDENT_ICAL_CACHE = HyperplanningManager.class.getName() + "$studentsIcals";
    private static final String __CAS_IDENTIFIER_CACHE = HyperplanningManager.class.getName() + "$casIdentifiers";
    
    /* Reduce the set of retrieved data to avoid the modifiant field that is wrongly defined in swagger */
    private static final List<String> __COURS_ANNULES_SELECT = List.of("cle", "matiere", "commentaire", "motif_annulation", "date_heure_annulation");
    
    private String _connectionLogin;
    private String _connectionPass;
    private String _serverUrl;
    
    private IcalsApi _icalApi;
    private CoursApi _coursApi;
    private CoursAnnulesApi _coursAnnulesApi;
    private EtudiantsApi _etudiantsApi;
    private MatieresApi _matiereApi;
    
    private AbstractCacheManager _cacheManager;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _connectionLogin = Config.getInstance().getValue("org.ametys.plugins.hyperplanning.login");
        _connectionPass = Config.getInstance().getValue("org.ametys.plugins.hyperplanning.password");
        _serverUrl = HttpUtils.sanitize(Config.getInstance().getValue("org.ametys.plugins.hyperplanning.url"));
        
        _cacheManager.createMemoryCache(__CANCELLED_LESSONS_CACHE,
                new I18nizableText("plugin.hyperplanning", "PLUGIN_HYPERPLANNING_CANCELLED_LESSONS_CACHE_LABEL"),
                new I18nizableText("plugin.hyperplanning", "PLUGIN_HYPERPLANNING_CANCELLED_LESSONS_CACHE_DESCRIPTION"),
                true,
                Duration.ofMinutes(Config.getInstance().getValue("org.ametys.plugins.hyperplanning.cache-validity")));
        
        _cacheManager.createMemoryCache(__STUDENT_ICAL_CACHE,
                new I18nizableText("plugin.hyperplanning", "PLUGIN_HYPERPLANNING_STUDENT_ICALS_CACHE_LABEL"),
                new I18nizableText("plugin.hyperplanning", "PLUGIN_HYPERPLANNING_STUDENT_ICALS_CACHE_DESCRIPTION"),
                true,
                Duration.ofMinutes(Config.getInstance().getValue("org.ametys.plugins.hyperplanning.cache-validity")));
        
        _cacheManager.createMemoryCache(__CAS_IDENTIFIER_CACHE,
                new I18nizableText("plugin.hyperplanning", "PLUGIN_HYPERPLANNING_CAS_IDENTIFIER_CACHE_LABEL"),
                new I18nizableText("plugin.hyperplanning", "PLUGIN_HYPERPLANNING_CAS_IDENTIFIER_CACHE_DESCRIPTION"),
                true,
                Duration.ofMinutes(Config.getInstance().getValue("org.ametys.plugins.hyperplanning.cache-validity")));
        
        _initializeWebService();
    }
    
    /**
     * Get the timetable of a student
     * @param userIdentity the student identity
     * @return the timetable as an ICS string
     */
    public String getStudentIcal(UserIdentity userIdentity)
    {
        Cache<UserIdentity, String> cache = _cacheManager.get(__STUDENT_ICAL_CACHE);
        return cache.get(userIdentity, this::_getStudentIcal);
    }
    
    
    /**
     * Get the timetable of a student
     * @param userIdentity the student identity
     * @return the timetable as an ICS string
     */
    protected String _getStudentIcal(UserIdentity userIdentity)
    {
        try
        {
            Optional<String> etudiantCle = getEtudiantCle(userIdentity);
            if (etudiantCle.isEmpty())
            {
                return null;
            }
            return _icalApi.icalEtudiantsClePost(etudiantCle.get(), null).getIcal();
        }
        catch (ApiException e)
        {
            getLogger().error("Failed to retrieve Ical for user", e);
            return null;
        }
    }
    
    /**
     * Get the list of cancelled lessons for a given user in the next 2 weeks
     * @param userIdentity the user identity
     * @return a list of {@link CancelledLesson} representing the cancelled lessons or null if the user is unknown
     * @throws UnknownStudentException when user is not linked to hyperplanning
     */
    public Set<CancelledLesson> getUpcomingCancelledLessons(UserIdentity userIdentity) throws UnknownStudentException
    {
        if (userIdentity == null)
        {
            throw new IllegalArgumentException("User is not connected");
        }
        
        Set<CancelledLesson> cancelledLessons = getCancelledLessonsCache().get(userIdentity, this::_getCancelledLessons);
        if (cancelledLessons == null)
        {
            throw new UnknownStudentException("User '" + userIdentity + "' has no link hyperplanning id");
        }
        
        return cancelledLessons;
    }
    
    /**
     * Get the list of cancelled lessons for a given user in the next 2 weeks
     * @param userIdentity the user identity
     * @return a list of {@link CancelledLesson} representing the cancelled lessons or null if the user is not linked to hyperplanning
     */
    private Set<CancelledLesson> _getCancelledLessons(UserIdentity userIdentity)
    {
        try
        {
            Optional<String> hypIdentity = getEtudiantCle(userIdentity);
            if (hypIdentity.isEmpty())
            {
                return null;
            }
            
            ZonedDateTime startDate = ZonedDateTime.now().truncatedTo(ChronoUnit.DAYS);
            ZonedDateTime endDate = startDate.plusWeeks(2).plusDays(1); // add one day to 'compensate' the rounding
            
            List<CoursAnnules> coursAnnules = _coursAnnulesApi.coursAnnulesGet(
                    null, /* sort */
                    __COURS_ANNULES_SELECT, /* select */
                    null, /*cle*/
                    null, /*matiere*/
                    null, /* type */
                    null, /* reference */
                    startDate.format(DateTimeFormatter.ISO_LOCAL_DATE_TIME), /* start */
                    endDate.format(DateTimeFormatter.ISO_LOCAL_DATE_TIME), /* end */
                    null, /* enseignants */
                    null, /* promotions */
                    null, /* tdoption */
                    null, /* regroupements */
                    null, /* salles */
                    List.of(hypIdentity.get())); /*etudiants*/
            
            Comparator<CancelledLesson> comparator = (c1, c2) -> c1.date().compareTo(c2.date());
            Set<CancelledLesson> result = new TreeSet<>(comparator);
            
            for (CoursAnnules cours: coursAnnules)
            {
                String motifAnnulation = cours.getMotifAnnulation();
                Integer cleMatiere = cours.getMatiere();
                // fetch the matiere once and for all
                Matieres matiere = _matiereApi.matieresCleGet(cleMatiere.toString());
                
                // A CoursAnnules contains a list of Seances
                // Fetch this list and restrict it to our time frame
                List<CoursAnnulesCleDetailSeancesPlaceesGet200ResponseInner> seances = _coursAnnulesApi.coursAnnulesCleDetailSeancesPlaceesGet(cours.getCle());
                for (CoursAnnulesCleDetailSeancesPlaceesGet200ResponseInner seance : seances)
                {
                    // Hyperplanning don't have a concept of time zone --"
                    // use the default time zone and hope for the best
                    ZonedDateTime dateSeance = LocalDateTime.parse(seance.getJourHeureDebut()).atZone(ZoneId.systemDefault());
                    if (dateSeance.isAfter(startDate) && dateSeance.isBefore(endDate))
                    {
                        result.add(new CancelledLesson(
                                matiere.getCode(),
                                matiere.getLibelle(),
                                matiere.getLibelleLong(),
                                dateSeance,
                                motifAnnulation,
                                cours.getCommentaire()
                            ));
                    }
                }
            }
            
            return result;
        }
        catch (ApiException e)
        {
            getLogger().error("An error occured while contacting Hyperplanning", e);
            return null;
        }
    }
    
    /**
     * Get the list of cancelled lessons and the impacted students.
     * @param cancellationMinDate a date to only return lessons that were cancelled after that date, or null.
     * @param lessonMinDate a date to only return lessons that start after that date, or null.
     * @param lessonMaxDate a date to only return lessons that start before that date, or null.
     * @return a map of cancelled lessons with the impacted CAS identifier
     */
    public Map<CancelledLesson, List<String>> getCancelledLessons(LocalDateTime cancellationMinDate, LocalDateTime lessonMinDate, LocalDateTime lessonMaxDate)
    {
        Predicate<CoursAnnules> cancellationMinDatePredicate = _getCancellationDateFilter(cancellationMinDate);
        
        Map<CancelledLesson, List<String>> result = new HashMap<>();
        try
        {
            String minDate = lessonMinDate != null ? lessonMinDate.truncatedTo(ChronoUnit.SECONDS).format(DateTimeFormatter.ISO_DATE_TIME) : null;
            String maxDate = lessonMaxDate != null ? lessonMaxDate.truncatedTo(ChronoUnit.SECONDS).format(DateTimeFormatter.ISO_DATE_TIME) : null;
            List<CoursAnnules> coursAnnules = _coursAnnulesApi.coursAnnulesGet(null, __COURS_ANNULES_SELECT, null, null, null, null, minDate, maxDate, null, null, null, null, null, null);
            for (CoursAnnules coursAnnule : coursAnnules)
            {
                // Keep only cancellation that match the cancellation date filter
                if (cancellationMinDatePredicate.test(coursAnnule))
                {
                    Matieres matiere = _matiereApi.matieresCleGet(coursAnnule.getMatiere().toString());
                    
                    Cours cours = null;
                    // Get the cancelled lessons
                    List<CoursAnnulesCleDetailSeancesPlaceesGet200ResponseInner> seances = _coursAnnulesApi.coursAnnulesCleDetailSeancesPlaceesGet(coursAnnule.getCle());
                    for (CoursAnnulesCleDetailSeancesPlaceesGet200ResponseInner seance :seances)
                    {
                        // filter again to keep only lesson that occurs in the time frame
                        // (when multiple lessons for the same course are cancelled together, they all appears in this list.
                        LocalDateTime dateSeance = LocalDateTime.parse(seance.getJourHeureDebut(), DateTimeFormatter.ISO_DATE_TIME);
                        if ((lessonMinDate == null || dateSeance.isAfter(lessonMinDate))
                                && (lessonMaxDate == null || dateSeance.isBefore(lessonMaxDate)))
                        {
                            // all cancelled lessons should have the same course compute it once
                            if (cours == null)
                            {
                                Integer cleCours = seance.getCleCours();
                                if (cleCours != 0)
                                {
                                    cours = _coursApi.coursCleGet(cleCours.toString());
                                }
                                else
                                {
                                    // no course links to the cancelled course. Ignore
                                    break;
                                }
                            }
                            
                            List<String> casIdentifiers = _getCASIdentifiers(cours.getEtudiants());
                            CancelledLesson cancelledLesson = new CancelledLesson(
                                    matiere.getCode(),
                                    matiere.getLibelle(),
                                    matiere.getLibelleLong(),
                                    dateSeance.atZone(ZoneId.systemDefault()),
                                    coursAnnule.getMotifAnnulation(),
                                    coursAnnule.getCommentaire()
                                    );
                            
                            result.put(cancelledLesson, casIdentifiers);
                        }
                    }
                }
            }
        }
        catch (ApiException e)
        {
            getLogger().error("Failed to compute notification of cancelled course", e);
        }
        
        return result;
    }

    private List<String> _getCASIdentifiers(List<Integer> etudiant)
    {
        Cache<String, String> cache = _cacheManager.get(__CAS_IDENTIFIER_CACHE);
        return etudiant.stream()
            .map(cle -> cache.get(cle.toString(), c -> {
                // Do not discards every student because of an error
                try
                {
                    return _etudiantsApi.etudiantsCleGet(c).getCasIdentifiant();
                }
                catch (ApiException e)
                {
                    getLogger().error("Failed to fetch CAS identifiant for student key: " + c);
                    return null;
                }
            }))
            .filter(Objects::nonNull)
            .toList();
    }

    private Predicate<CoursAnnules> _getCancellationDateFilter(LocalDateTime cancellationMinDate)
    {
        return cancellationMinDate != null
                ? c -> LocalDateTime.parse(c.getDateHeureAnnulation(), DateTimeFormatter.ISO_DATE_TIME).isAfter(cancellationMinDate)
                : c -> true;
    }
    
    /**
     * Get the hyperplanning student key corresponding to the user identity
     * @param userIdentity the user identity
     * @return the cle or empty if the user doesn't match a hyperplanning student
     * @throws ApiException if an error occurs
     */
    protected Optional<String> getEtudiantCle(UserIdentity userIdentity) throws ApiException
    {
        List<Etudiants> etudiants = _etudiantsApi.etudiantsGet(
                null,
                List.of("cle"), /* select */
                null,
                null,
                null,
                null,
                null,
                null,
                null,
                List.of(userIdentity.getLogin()), /* CAS identity */
                null,
                null
        );
        
        if (etudiants.isEmpty())
        {
            getLogger().debug("No matching hyperplanning student for CAS login: " + userIdentity.getLogin());
            return Optional.empty();
        }
        else if (etudiants.size() > 1)
        {
            getLogger().info("Multiple hyperplanning student for CAS login: " + userIdentity.getLogin());
            return Optional.empty();
        }
        
        return Optional.of(etudiants.get(0).getCle().toString());
    }
    
    private void _initializeWebService()
    {
        ApiClient client = new ApiClient();
        client.setBasePath(_serverUrl + "/hpsw/api/v1");
        client.setUsername(_connectionLogin);
        client.setPassword(_connectionPass);
        
        _matiereApi = new MatieresApi(client);
        _coursApi = new CoursApi(client);
        _coursAnnulesApi = new CoursAnnulesApi(client);
        _etudiantsApi = new EtudiantsApi(client);
        _icalApi = new IcalsApi(client);
    }
    
    private Cache<UserIdentity, Set<CancelledLesson>> getCancelledLessonsCache()
    {
        return _cacheManager.get(__CANCELLED_LESSONS_CACHE);
    }
    
    /**
     * Represent a cancelled lesson from hyperplanning
     * @param code the code of the subject this lesson belongs to
     * @param label the label of the subject this lesson belongs to
     * @param fullLabel the full label of the subject this lesson belongs to
     * @param date the original date of the lesson
     * @param cancelRationale the cancellation rationale
     * @param cancelComment the cancellation comment
     */
    public record CancelledLesson(
        String code,
        String label,
        String fullLabel,
        ZonedDateTime date,
        String cancelRationale,
        String cancelComment
    ) { }

}
