/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.link;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Context;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.Binary;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.ProfileAssignmentStorageExtensionPoint;
import org.ametys.core.right.RightManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.upload.Upload;
import org.ametys.core.upload.UploadManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.linkdirectory.DirectoryEvents;
import org.ametys.plugins.linkdirectory.DirectoryHelper;
import org.ametys.plugins.linkdirectory.Link;
import org.ametys.plugins.linkdirectory.Link.LinkStatus;
import org.ametys.plugins.linkdirectory.Link.LinkType;
import org.ametys.plugins.linkdirectory.Link.LinkVisibility;
import org.ametys.plugins.linkdirectory.LinkDirectoryColorsComponent;
import org.ametys.plugins.linkdirectory.dynamic.DynamicInformationProvider;
import org.ametys.plugins.linkdirectory.dynamic.DynamicInformationProviderExtensionPoint;
import org.ametys.plugins.linkdirectory.repository.DefaultLink;
import org.ametys.plugins.linkdirectory.repository.DefaultLinkFactory;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * DAO for manipulating {@link Link}
 */
public class LinkDAO extends AbstractLogEnabled implements Serviceable, Component, Contextualizable
{
    /** Avalon Role */
    public static final String ROLE = LinkDAO.class.getName();
    
    private AmetysObjectResolver _resolver;
    
    private ObservationManager _observationManager;
    private SiteManager _siteManager;
    private CurrentUserProvider _currentUserProvider;
    private UploadManager _uploadManager;
    private RightManager _rightManager;
    private ProfileAssignmentStorageExtensionPoint _profileAssignmentStorageEP;

    private JSONUtils _jsonUtils;
    private DirectoryHelper _directoryHelper;
    
    private Context _cocoonContext;
    private org.apache.avalon.framework.context.Context _context;

    private DynamicInformationProviderExtensionPoint _dynamicInfoExtensionPoint;
    
    private LinkDirectoryColorsComponent _colorComponent;

    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _uploadManager = (UploadManager) manager.lookup(UploadManager.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _directoryHelper = (DirectoryHelper) manager.lookup(DirectoryHelper.ROLE);
        _dynamicInfoExtensionPoint = (DynamicInformationProviderExtensionPoint) manager.lookup(DynamicInformationProviderExtensionPoint.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _profileAssignmentStorageEP = (ProfileAssignmentStorageExtensionPoint) manager.lookup(ProfileAssignmentStorageExtensionPoint.ROLE);
        _colorComponent = (LinkDirectoryColorsComponent) manager.lookup(LinkDirectoryColorsComponent.ROLE);
    }
    
    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    /**
     * Create a new link
     * @param parameters a map of the following parameters for the link : siteName, language, url, title, content, url-alternative, picture, picture#type, picture-alternative, themes, grant-any-user, fousers, fogroups
     * @return The new link id
     */
    @Callable (rights = "LinkDirectory_Rights_Links_Handle")
    public Map<String, Object> createLink(Map<String, Object> parameters)
    {
        Map<String, Object> result = new HashMap<>();
        
        String siteName = (String) parameters.get("siteName");
        String language = (String) parameters.get("lang");
        Site site = _siteManager.getSite(siteName);

        String url = StringUtils.defaultString((String) parameters.get("url"));
        String internalUrl = StringUtils.defaultString((String) parameters.get("internal-url"));
        
        // Check that the word doesn't already exist.
        if (_urlExists(url, internalUrl, siteName, language))
        {
            result.put("already-exists", true);
            return result;
        }

        ModifiableTraversableAmetysObject rootNode = _directoryHelper.getLinksNode(site, language);
        
        String name = url;
        if (StringUtils.isBlank(name))
        {
            name = internalUrl;
        }
        DefaultLink link = _createLink(name, rootNode);
        _setValues(link, parameters);
        
        rootNode.saveChanges();
        
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, link.getId());
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, rootNode.getId());
        eventParams.put(ObservationConstants.ARGS_NAME, link.getName());
        eventParams.put(ObservationConstants.ARGS_PATH, link.getPath());

        _observationManager.notify(new Event(DirectoryEvents.LINK_CREATED, _currentUserProvider.getUser(), eventParams));
        
        // Set public access
        _setAccess(link, null);
        
        return convertLink2JsonObject(link);
    }
    
    /**
     * Create a new user link
     * @param parameters a map of the following parameters for the link : siteName, language, url, title, content, url-alternative, picture, picture#type, picture-alternative
     * @return The new link id
     */
    public Map<String, Object> createUserLink(Map<String, Object> parameters)
    {
        Map<String, Object> result = new HashMap<>();
        
        // Retrieve the current workspace.
        Request request = ContextHelper.getRequest(_context);
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        DefaultLink link = null;
        
        try
        {
            // Force default workspace for link creation (generally created from FO)
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            UserIdentity currentUser = _currentUserProvider.getUser();
            if (currentUser == null)
            {
                result.put("unauthenticated-user", true);
                return result;
            }
            
            String siteName = (String) parameters.get("siteName");
            String language = (String) parameters.get("lang");
            Site site = _siteManager.getSite(siteName);

            String url = StringUtils.defaultString((String) parameters.get("url"));
            String internalUrl = StringUtils.defaultString((String) parameters.get("internal-url"));

            // Check that the word doesn't already exist for the given user.
            if (_urlExistsForUser(url, internalUrl, siteName, language, currentUser))
            {
                result.put("already-exists", true);
                return result;
            }
            
            ModifiableTraversableAmetysObject rootNodeForUser = _directoryHelper.getLinksForUserNode(site, language, currentUser);
            
            String name = url;
            if (StringUtils.isBlank(name))
            {
                name = internalUrl;
            }
            
            link = _createLink(name, rootNodeForUser);
            _setValues(link, parameters);
            
            rootNodeForUser.saveChanges();

            // Notify listeners
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_ID, link.getId());
            eventParams.put(ObservationConstants.ARGS_PARENT_ID, rootNodeForUser.getId());
            eventParams.put(ObservationConstants.ARGS_NAME, link.getName());
            eventParams.put(ObservationConstants.ARGS_PATH, link.getPath());

            _observationManager.notify(new Event(DirectoryEvents.LINK_CREATED, _currentUserProvider.getUser(), eventParams));
            
            // Set public access
            _setAccess(link, currentUser);
        }
        finally
        {
            // Restore context before getting link data
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
        
        // At this point link cannot be null
        return convertLink2JsonObject(link);
    }
    
    /**
     * Indicate if a link is a user link
     * @param link the link
     * @return true if the link is a user link
     */
    public boolean isUserLink(DefaultLink link)
    {
        AmetysObject parent = link.getParent();
        AmetysObject linksNode = _directoryHelper.getLinksNode(link.getSite(), link.getLanguage());
        return !parent.equals(linksNode);
    }

    
    /**
     * Updates a link
     * @param parameters a map of the following parameters for the link : siteName, language, id, url, title, content, url-alternative, picture, picture#type, picture-alternative, themes, grant-any-user, fousers, fogroups
     * @return the update link
     */
    @Callable (rights = "LinkDirectory_Rights_Links_Handle")
    public Map<String, Object> updateLink(Map<String, Object> parameters)
    {
        Map<String, Object> result = new HashMap<>();
        
        String siteName = (String) parameters.get("siteName");
        String language = (String) parameters.get("language");
        String id = StringUtils.defaultString((String) parameters.get("id"));
        String url = StringUtils.defaultString((String) parameters.get("url"));
        String internalUrl = StringUtils.defaultString((String) parameters.get("internal-url"));
        
        try
        {
            DefaultLink link = _resolver.resolveById(id);
            
            // If the url was changed, check that the new url doesn't already exist.
            if (!link.getUrl().equals(url) && _urlExists(url, internalUrl, siteName, language))
            {
                result.put("already-exists", true);
                return result;
            }
            
            _setValues(link, parameters);
            link.saveChanges();
            
            // Notify listeners
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_ID, link.getId());
            eventParams.put(ObservationConstants.ARGS_NAME, link.getName());
            eventParams.put(ObservationConstants.ARGS_PATH, link.getPath());
    
            _observationManager.notify(new Event(DirectoryEvents.LINK_MODIFIED, _currentUserProvider.getUser(), eventParams));
            
            return convertLink2JsonObject(link);
        }
        catch (UnknownAmetysObjectException e)
        {
            result.put("unknown-link", true);
            return result;
        }
        catch (AmetysRepositoryException e)
        {
            throw new IllegalStateException(e);
        }
    }
    
    @SuppressWarnings("unchecked")
    private void _setValues(Link link, Map<String, Object> values)
    {
        // Set values (url, type, title, etc.)
        _setLinkValues(link, values);
        
        // Set themes
        List<String> themes = Collections.EMPTY_LIST;
        Object themesFromValues = values.get("themes");
        if (themesFromValues instanceof List)
        {
            themes = (List<String>) themesFromValues;
        }
        else if (themesFromValues instanceof String)
        {
            themes = _jsonUtils.convertJsonToList((String) themesFromValues).stream()
                         .map(Object::toString)
                         .collect(Collectors.toList());
        }

        _setThemes(link, themes);
    }
    
    private void _setLinkValues(Link link, Map<String, Object> values)
    {
        String url = StringUtils.defaultString((String) values.get("url"));
        String dynInfoProviderId = StringUtils.defaultString((String) values.get("dynamic-info-provider"));
        String internalUrl = StringUtils.defaultString((String) values.get("internal-url"));
        String urlType = StringUtils.defaultString((String) values.get("url-type"));
        String title = StringUtils.defaultString((String) values.get("title"));
        String content = StringUtils.defaultString((String) values.get("content"));
        String alternative = StringUtils.defaultString((String) values.get("url-alternative"));
        String color = StringUtils.defaultString((String) values.get("color"));
        String pageId = StringUtils.defaultString((String) values.get("page"));
        String status = StringUtils.defaultString((String) values.get("status"));
        String defaultVisibility = StringUtils.defaultString((String) values.get("default-visibility"));
        
        String pictureAsStr = StringUtils.defaultString((String) values.get("picture"));
        String pictureAlternative = StringUtils.defaultString((String) values.get("picture-alternative"));

        // Check the dynamic provider still exists
        if (!_dynamicInfoExtensionPoint.hasExtension(dynInfoProviderId))
        {
            dynInfoProviderId = "";
        }

        link.setUrl(LinkType.valueOf(urlType), url);
        link.setDynamicInformationProvider(dynInfoProviderId);
        link.setInternalUrl(internalUrl);
        link.setTitle(title);
        link.setContent(content);
        link.setAlternative(alternative);
        link.setPictureAlternative(pictureAlternative);
        link.setColor(color);
        link.setPage(pageId);
        link.setStatus(StringUtils.isNotBlank(status) ? LinkStatus.valueOf(status) : LinkStatus.NORMAL);
        link.setDefaultVisibility(StringUtils.isNotBlank(defaultVisibility) ? LinkVisibility.valueOf(defaultVisibility) : LinkVisibility.VISIBLE);
        
        _setPicture(link, pictureAsStr);
    }
    
    private void _setThemes(Link link, List<String> themes)
    {
        link.setThemes(themes.toArray(new String[themes.size()]));
    }
    
    private void _setPicture(Link link, String valueAsStr)
    {
        if (StringUtils.isNotEmpty(valueAsStr))
        {
            Map<String, Object> picture = _jsonUtils.convertJsonToMap(valueAsStr);
            
            if (!picture.isEmpty())
            {
                String pictureType = (String) picture.get("type");
                String value = (String) picture.get("id");
                
                if (pictureType.equals("explorer") && !"untouched".equals(value))
                {
                    link.setResourcePicture(value);
                }
                else if (pictureType.equals("glyph"))
                {
                    link.setPictureGlyph(value);
                }
                else if (!"untouched".equals(value))
                {
                    UserIdentity user = _currentUserProvider.getUser();
                    Upload upload = _uploadManager.getUpload(user, value);
                    
                    String filename = upload.getFilename();
                    String mimeType = upload.getMimeType() != null ? upload.getMimeType() : _cocoonContext.getMimeType(filename);
                    String finalMimeType = mimeType != null ? mimeType : "application/unknown";
                    
                    link.setExternalPicture(finalMimeType, filename, upload.getInputStream());
                }
            }
            else
            {
                // Remove picture
                link.setNoPicture();
            }
            
        }
        else
        {
            // Remove picture
            link.setNoPicture();
        }
    }
    
    /**
     * Delete one or multiples links
     * @param ids a list of links' ids
     * @return true if all the links were deleted, false if at least one link could not be delete.
     */
    @Callable (rights = "LinkDirectory_Rights_Links_Handle")
    public List<String> deleteLinks(List<String> ids)
    {
        List<String> result = new ArrayList<>();
        
        for (String id : ids)
        {
            try
            {
                DefaultLink link = _resolver.resolveById(id);
                
                deleteLink(link);
    
                result.add(id);
            }
            catch (UnknownAmetysObjectException e)
            {
                // ignore the id and continue the deletion.
                getLogger().error("Unable to delete the link of id '" + id + ", because it does not exist.", e);
            }
        }
        
        return result;
    }

    /**
     * Delete a link
     * @param link the link
     * @throws AmetysRepositoryException if an error occurred
     */
    public void deleteLink(DefaultLink link) throws AmetysRepositoryException
    {
        String siteName = link.getSiteName();
        String language = link.getLanguage();
        String linkId = link.getId();
        String name = link.getName();
        String path = link.getPath();
        
        ModifiableAmetysObject parent = link.getParent();
        link.remove();
   
        parent.saveChanges();
   
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, linkId);
        eventParams.put(ObservationConstants.ARGS_NAME, name);
        eventParams.put(ObservationConstants.ARGS_PATH, path);
        eventParams.put("siteName", siteName);
        eventParams.put("language", language);
        _observationManager.notify(new Event(DirectoryEvents.LINK_DELETED, _currentUserProvider.getUser(), eventParams));
    }
    
    /**
     * Move a link in the list
     * @param id the link id
     * @param role the move action
     * @throws RepositoryException if a repository error occurs.
     * @return the moved link in JSON
     */
    @Callable (rights = "LinkDirectory_Rights_Links_Handle")
    public Map<String, Object> moveLink(String id, String role) throws RepositoryException
    {
        DefaultLink link = _resolver.resolveById(id);
        
        switch (role)
        {
            case "move-first":
                _moveFirst(link);
                break;
            case "move-up":
                _moveUp(link);
                break;
            case "move-down":
                _moveDown(link);
                break;
            case "move-last":
                _moveLast(link);
                break;
            default:
                break;
        }
        
        return convertLink2JsonObject(link);
    }
    
    
    
    /**
     * Get the JSON object representing a link
     * @param id the id of link
     * @return the link as JSON object
     */
    @Callable (rights = "LinkDirectory_Rights_Links_Handle")
    public Map<String, Object> getLink (String id)
    {
        DefaultLink link = _resolver.resolveById(id);
        return convertLink2JsonObject(link);
    }
    
    /**
     * Determines if the restriction IP parameter is not empty
     * @param siteName the site name
     * @return true if the restriction IP parameter is not empty
     */
    @Callable (rights = "LinkDirectory_Rights_Links_Handle")
    public boolean isInternalURLAllowed (String siteName)
    {
        Site site = _siteManager.getSite(siteName);
        String allowedIpParameter = site.getValue("allowed-ip");
        return StringUtils.isNotBlank(allowedIpParameter);
    }
    
    /**
     * Returns the list of providers of dynamic information as json object
     * @return the providers of dynamic information
     */
    @Callable (rights = "LinkDirectory_Rights_Links_Handle")
    public List<Map<String, Object>> getDynamicInformationProviders()
    {
        List<Map<String, Object>> result = new ArrayList<>();
        
        for (String id : _dynamicInfoExtensionPoint.getExtensionsIds())
        {
            DynamicInformationProvider provider = _dynamicInfoExtensionPoint.getExtension(id);
            Map<String, Object> info = new HashMap<>();
            info.put("id", provider.getId());
            info.put("label", provider.getLabel());
            result.add(info);
        }
        
        return result;
    }
    
    /**
     * Convert a link to JSON object
     * @param link the link
     * @return the link as JSON object
     */
    public Map<String, Object> convertLink2JsonObject (DefaultLink link)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", link.getId());
        infos.put("lang", link.getLanguage());
        infos.put("url", link.getUrl());
        infos.put("dynamicInfoProvider", link.getDynamicInformationProvider());
        infos.put("internalUrl", link.getInternalUrl());
        infos.put("urlType", link.getUrlType().toString());
        
        if (link.getUrlType() == LinkType.PAGE)
        {
            String pageId = link.getUrl();
            try
            {
                Page page = _resolver.resolveById(pageId);
                infos.put("pageTitle", page.getTitle());
            }
            catch (UnknownAmetysObjectException e)
            {
                infos.put("unknownPage", true);
            }
        }
        
        infos.put("title", link.getTitle());
        infos.put("alternative", link.getAlternative());
        
        infos.put("content", StringUtils.defaultString(link.getContent()));
        infos.put("pictureAlternative", StringUtils.defaultString(link.getPictureAlternative()));
        
        Map<String, Object> pictureInfos = new HashMap<>();
        String pictureType = link.getPictureType();
        
        TraversableAmetysObject parent = link.getParent();
        infos.put("position", parent.getChildPosition(link));
        infos.put("count", parent.getChildren().getSize());
        
        infos.put("color", StringUtils.defaultString(link.getColor()));
        infos.put("page", StringUtils.defaultString(link.getPage()));
        if (link.getStatus() != null)
        {
            infos.put("status", link.getStatus().name());
        }
        infos.put("defaultVisibility", link.getDefaultVisibility().name());
        
        if (pictureType.equals("resource"))
        {
            String resourceId = link.getResourcePictureId();
            pictureInfos.put("id", resourceId);
            try
            {
                Resource resource = _resolver.resolveById(resourceId);
                
                pictureInfos.put("filename", resource.getName());
                pictureInfos.put("size", resource.getLength());
                pictureInfos.put("type", "explorer");
                pictureInfos.put("lastModified", resource.getLastModified());
                
                String viewUrl = ResolveURIComponent.resolve("explorer", resourceId, false);
                String downloadUrl = ResolveURIComponent.resolve("explorer", resourceId, true);
                pictureInfos.put("viewUrl", viewUrl);
                pictureInfos.put("downloadUrl", downloadUrl);
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().error("The resource of id'" + resourceId + "' does not exist anymore. The picture for link of id '" + link.getId() + "' will be ignored.", e);
                infos.put("pictureNotFound", true);
            }
        }
        else if (pictureType.equals("external"))
        {
            Binary picMeta = link.getExternalPicture();
            
            pictureInfos.put("path", DefaultLink.PROPERTY_PICTURE);
            pictureInfos.put("filename", picMeta.getFilename());
            pictureInfos.put("size", picMeta.getLength());
            pictureInfos.put("lastModified", picMeta.getLastModificationDate());
            pictureInfos.put("type", "link-data");
            
            String viewUrl = ResolveURIComponent.resolve("link-data", DefaultLink.PROPERTY_PICTURE + "?objectId=" + link.getId(), false);
            String downloadUrl = ResolveURIComponent.resolve("link-data", DefaultLink.PROPERTY_PICTURE + "?objectId=" + link.getId(), true);
            
            pictureInfos.put("viewUrl", viewUrl);
            pictureInfos.put("downloadUrl", downloadUrl);
            
        }
        else if (pictureType.equals("glyph"))
        {
            pictureInfos.put("id", link.getPictureGlyph());
            pictureInfos.put("type", "glyph");
        }
        infos.put("picture", pictureInfos);
        
        infos.put("isRestricted", !_rightManager.hasAnonymousReadAccess(link));
        
        // Themes
        List<Map<String, Object>> themesList = new ArrayList<>();
        for (String themeId : link.getThemes())
        {
            try
            {
                I18nizableText themeTitle = _directoryHelper.getThemeTitle(themeId, link.getSiteName(), link.getLanguage());
                Map<String, Object> themeData = new HashMap<>();
                themeData.put("id", themeId);
                themeData.put("label", themeTitle);
                themesList.add(themeData);
            }
            catch (UnknownAmetysObjectException e)
            {
                // Theme does not exist anymore
            }
        }
        
        infos.put("themes", themesList);
        
        return infos;
    }
    
    /**
     * Get links infos
     * @param linkIds the link id
     * @return the link infos
     */
    @Callable (rights = "LinkDirectory_Rights_Links_Handle")
    public List<Map<String, Object>> getLinks(List<String> linkIds)
    {
        List<Map<String, Object>> result = new ArrayList<>();
        
        for (String linkId: linkIds)
        {
            try
            {
                result.add(getLink(linkId));
            }
            catch (UnknownAmetysObjectException e)
            {
                // does not exists
            }
        }
        return result;
    }
    
    /**
     * Test if a link with the specified url or internal url exists in the directory.
     * @param url the url to test.
     * @param internalUrl the internal url to test
     * @param siteName the site name.
     * @param language the language.
     * @return true if the link exists.
     * @throws AmetysRepositoryException if a repository error occurs.
     */
    protected boolean _urlExists(String url, String internalUrl, String siteName, String language) throws AmetysRepositoryException
    {
        boolean externalLinkExists = false;
        if (StringUtils.isNotBlank(url))
        {
            String externalLinkXpathQuery = _directoryHelper.getUrlExistsQuery(siteName, language, url);
            try (AmetysObjectIterable<DefaultLink> externalLinks = _resolver.query(externalLinkXpathQuery);)
            {
                externalLinkExists = externalLinks.iterator().hasNext();
            }
        }
        
        boolean internalLinkExists = false;
        if (StringUtils.isNotBlank(internalUrl))
        {
            String internalLinkXpathQuery = _directoryHelper.getUrlExistsQuery(siteName, language, internalUrl);
            try (AmetysObjectIterable<DefaultLink> internalLinks = _resolver.query(internalLinkXpathQuery);)
            {
                internalLinkExists = internalLinks.iterator().hasNext();
            }
        }
        
        return externalLinkExists || internalLinkExists;
    }
    
    /**
     * Test if a link with the specified url or internal url exists in the directory for the given user.
     * @param url the url to test.
     * @param internalUrl the internal url to test
     * @param siteName the site name.
     * @param language the language.
     * @param user The user identity
     * @return true if the link exists for the given user.
     * @throws AmetysRepositoryException if a repository error occurs.
     */
    protected boolean _urlExistsForUser(String url, String internalUrl, String siteName, String language, UserIdentity user) throws AmetysRepositoryException
    {
        boolean externalLinkExists = false;
        if (StringUtils.isNotBlank(url))
        {
            String externalLinkXpathQuery = _directoryHelper.getUrlExistsForUserQuery(siteName, language, url, user);
            try (AmetysObjectIterable<DefaultLink> externalLinks = _resolver.query(externalLinkXpathQuery);)
            {
                externalLinkExists = externalLinks.iterator().hasNext();
            }
        }
        
        boolean internalLinkExists = false;
        if (StringUtils.isNotBlank(internalUrl))
        {
            String internalLinkXpathQuery = _directoryHelper.getUrlExistsForUserQuery(siteName, language, internalUrl, user);
            try (AmetysObjectIterable<DefaultLink> internalLinks = _resolver.query(internalLinkXpathQuery);)
            {
                internalLinkExists = internalLinks.iterator().hasNext();
            }
        }
        
        return externalLinkExists || internalLinkExists;
    }
    
    /**
     * Create the link object.
     * @param name the desired link name.
     * @param rootNode the links root node.
     * @return the created Link.
     */
    protected DefaultLink _createLink(String name, ModifiableTraversableAmetysObject rootNode)
    {
        String originalName = NameHelper.filterName(name);
        
        // Find unique name
        String uniqueName = originalName;
        int index = 2;
        while (rootNode.hasChild(uniqueName))
        {
            uniqueName = originalName + "-" + (index++);
        }
        
        return rootNode.createChild(uniqueName, DefaultLinkFactory.LINK_NODE_TYPE);
    }
    
    /**
     * Move link the first position
     * @param link the link to move
     * @throws RepositoryException if an error occurs while exploring the repository
     */
    private void _moveFirst(DefaultLink link) throws RepositoryException
    {
        try (AmetysObjectIterable<AmetysObject>  children = ((TraversableAmetysObject) link.getParent()).getChildren();)
        {
            // Resolve the link in the same session or the linkRoot.saveChanges() call below won't see the order changes.
            link.orderBefore(((TraversableAmetysObject) link.getParent()).getChildren().iterator().next());
            ((ModifiableAmetysObject) link.getParent()).saveChanges();
        }
    }

    /**
     * Move link after its following
     * @param link the link to move down
     * @throws RepositoryException if an error occurs while exploring the repository
     */
    private void _moveDown(DefaultLink link) throws RepositoryException
    {
        TraversableAmetysObject parentNode = link.getParent();
        AmetysObjectIterable<AmetysObject> siblings = parentNode.getChildren();
        Iterator<AmetysObject> it = siblings.iterator();
        
        boolean iterate = true;
        while (it.hasNext() && iterate)
        {
            DefaultLink sibling = (DefaultLink) it.next();
            iterate = !sibling.getName().equals(link.getName());
        }
        
        if (it.hasNext())
        {
            // Move the link after his next sibling: move the next sibling before the link to move.
            DefaultLink nextLink = (DefaultLink) it.next();
            nextLink.orderBefore(link);
    
            link.saveChanges();
        }
    }
    
    /**
     * Move link before its preceding
     * @param link the link to move up
     * @throws RepositoryException if an error occurs while exploring the repository
     */
    private void _moveUp(DefaultLink link) throws RepositoryException
    {
        TraversableAmetysObject parentNode = link.getParent();
        AmetysObjectIterable<AmetysObject> siblings = parentNode.getChildren();
        Iterator<AmetysObject> it = siblings.iterator();
        DefaultLink previousLink = null;

        while (it.hasNext())
        {
            DefaultLink sibling = (DefaultLink) it.next();
            if (sibling.getName().equals(link.getName()))
            {
                break;
            }
            
            previousLink = sibling;
        }
            
        if (previousLink != null)
        {
            // Move the link after his next sibling: move the next sibling before the link to move.
            link.orderBefore(previousLink);
            link.saveChanges();
        }
    }
    
    /**
     * Move link to the last position.
     * @param link the link to move up
     * @throws RepositoryException if an error occurs while exploring the repository
     */
    private void _moveLast(DefaultLink link) throws RepositoryException
    {
        link.moveTo(link.getParent(), false);
        ((ModifiableAmetysObject) link.getParent()).saveChanges();
    }
    
    /**
     * Set access to the link
     * @param link the link
     * @param user the user to set access. Can be null, in this case, we set anonymous right
     */
    private void _setAccess (Link link, UserIdentity user)
    {
        if (user != null)
        {
            _profileAssignmentStorageEP.allowProfileToUser(user, RightManager.READER_PROFILE_ID, link);
        }
        else
        {
            _profileAssignmentStorageEP.allowProfileToAnonymous(RightManager.READER_PROFILE_ID, link);
        }
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_CONTEXT, link);
        eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_PROFILES, Collections.singleton(RightManager.READER_PROFILE_ID));
        
        _observationManager.notify(new Event(org.ametys.core.ObservationConstants.EVENT_ACL_UPDATED, _currentUserProvider.getUser(), eventParams));
    }
    
    /**
     * Retrieves the color of the link.
     * If there is no color configured on the link, the default site color
     * @param link the link
     * @return the color of the link
     */
    public String getLinkColor(DefaultLink link)
    {
        Map<String, Map<String, String>> colors = _colorComponent.getColors(link.getSiteName());
        if (colors.containsKey(link.getColor()))
        {
            return colors.get(link.getColor()).get("main");
        }
        else
        {
            return colors.get(_colorComponent.getDefaultKey(link.getSiteName())).get("main");
        }
    }
}
