/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The object that execute actions on links
 */
Ext.define("Ametys.plugins.linkdirectory.link.LinkDirectoryActions", {
    singleton: true,
   
    /**
     * @private
     * @property __UITOOL_LINK_DIRECTORY The id of link directory UI tool
     */
    __UITOOL_LINK_DIRECTORY: 'uitool-linkdirectory',
    
    /**
     * Creates a new link to an external url
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
     */
    addLink: function(controller)
    {
    	var tool = Ametys.tool.ToolsManager.getTool(this.__UITOOL_LINK_DIRECTORY);
 		var lang = tool != null ? tool.getCurrentLanguage() : Ametys.cms.language.LanguageDAO.getCurrentLanguage();
 		
 		Ametys.plugins.linkdirectory.link.EditLinkDialog.open ({linkType: 'URL', lang: lang, callback: Ext.bind(this._addLinkCb, this)});
    },
    
    /**
     * Creates a new link to a internal page
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
     */
    addPage: function(controller)
    {
        var tool = Ametys.tool.ToolsManager.getTool(this.__UITOOL_LINK_DIRECTORY);
        var lang = tool != null ? tool.getCurrentLanguage() : Ametys.cms.language.LanguageDAO.getCurrentLanguage();
        
        Ametys.plugins.linkdirectory.link.EditLinkDialog.open ({linkType: 'PAGE', lang: lang, callback: Ext.bind(this._addLinkCb, this)});
    },
    
    /**
     * @private
     * Callback function after adding the new link
     * Open the UI tool if it is not yet opened.
     * @param {String} id The id of created link
     */
    _addLinkCb: function (id)
    {
		var tool = Ametys.tool.ToolsManager.getTool(this.__UITOOL_LINK_DIRECTORY);
    	if (tool == null)
    	{
    		Ametys.tool.ToolsManager.openTool(this.__UITOOL_LINK_DIRECTORY, {selectedLinkIds: [id]});
    	}
    },
    
    /**
 	 * Edit the selected link
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	editLink: function (controller)
 	{
 		var targets = Ametys.message.MessageBus.getCurrentSelectionMessage().getTargets();
		var target = Ametys.message.MessageTargetHelper.findTarget(targets, function(target) { return target.getId() == Ametys.message.MessageTarget.LINK_DIRECTORY; });
		
		if (target == null)
		{
			return;
		}
			
		var linkId = target.getParameters().id;
		var tool = Ametys.tool.ToolsManager.getTool(this.__UITOOL_LINK_DIRECTORY);
 		var lang = tool != null ? tool.getCurrentLanguage() : Ametys.cms.language.LanguageDAO.getCurrentLanguage();
 		
 		Ametys.plugins.linkdirectory.link.EditLinkDialog.open ({
 			mode: 'edit', 
 			id: linkId, 
 			lang: lang
 		});
 	},
 	
 	/**
 	 * Delete the selected link
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	deleteLink: function(controller)
 	{
 		Ametys.Msg.confirm("{{i18n PLUGINS_LINKDIRECTORY_LINK_DELETE_TITLE}}",
 				"{{i18n PLUGINS_LINKDIRECTORY_LINK_DELETE_CONFIRM}}",
 				this._doDelete,
 				this
 		);
 	},
 	 	
 	/**
 	 * @private
 	 * The action to perform when the user clicks on a button from the deleting message box.
 	 * @param {String} btn The pressed button. Can only be 'yes'/'no'
 	 */
 	_doDelete: function(btn)
 	{
	 	if (btn == 'yes')
	    {
	        var targets = Ametys.message.MessageBus.getCurrentSelectionMessage().getTargets();
	        var allTargets = Ametys.message.MessageTargetHelper.findTargets(targets, function(target) { return target.getId() == Ametys.message.MessageTarget.LINK_DIRECTORY; });
	        
	        if (allTargets.length == 0)
	        {
	            return;
	        }
	        
	        var ids = [];
	        for (var i = 0; i < allTargets.length; i++)
	        {
	            ids.push(allTargets[i].getParameters().id);
	        } 
	        
	        var tool = Ametys.tool.ToolsManager.getFocusedTool();
	        Ametys.plugins.linkdirectory.link.LinkDAO.deleteLinks([ids, targets], null, { waitMessage: {
                target: tool.getContentPanel(),
                msg: "{{i18n PLUGINS_LINKDIRECTORY_LINK_REMOVING_WAIT}}"
            }});
	    }
 	},
 	

	/**
 	 * Move the selected link
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 * @param {String} controller.role The mode for move. Accepted values are: 'up', 'down', 'first' or 'last'.
 	 */
    moveLink: function(controller)
    {
        var targets = controller.getMatchingTargets();          
        if (targets.length > 0)
        {
        	var id = targets[0].getParameters().id;
            Ametys.plugins.linkdirectory.link.LinkDAO.moveLink([id, controller.role], null, {});
        }
    }
});
