/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.XMLUtils;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.dom.AmetysNodeList;
import org.ametys.core.util.dom.MapElement;
import org.ametys.plugins.linkdirectory.repository.DefaultLink;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.transformation.xslt.AmetysXSLTHelper;

/**
 * XSLT Helper with link directory specificity
 */
public class LinkDirectoryXSLTHelper extends AmetysXSLTHelper
{
    private static LinkDirectoryColorsComponent _linkDirectoryColorsComponent;
    private static DirectoryHelper _directoryHelper;
    private static SiteManager _siteManager;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _linkDirectoryColorsComponent = (LinkDirectoryColorsComponent) manager.lookup(LinkDirectoryColorsComponent.ROLE);
        _directoryHelper = (DirectoryHelper) manager.lookup(DirectoryHelper.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }
    
    /**
     * Get all the colors for the current site
     * @return all colors available for the current site
     */
    public static MapElement getColors()
    {
        String siteName = site();
        return getColors(siteName);
    }
    
    /**
     * Get all the colors for a site
     * @param siteName site to check
     * @return all colors available for this site
     */
    public static MapElement getColors(String siteName)
    {
        Map<String, Map<String, String>> colors = _linkDirectoryColorsComponent.getColors(siteName);
        
        return new MapElement("colors", colors);
    }
    
    /**
     * Get the default color index for the current site
     * @return the default color index for the current site
     */
    public static String getDefaultColorIndex()
    {
        String siteName = site();
        return getDefaultColorIndex(siteName);
    }
    
    /**
     * Get the default color index for a site
     * @param siteName site to check
     * @return the default color index for this site
     */
    public static String getDefaultColorIndex(String siteName)
    {
        return _linkDirectoryColorsComponent.getDefaultKey(siteName);
    }
    
    /**
     * Get the links that belong to a given theme. Only links allowed to anonymous access are returned !
     * Internal URL are ignored.
     * @param siteName the site name
     * @param lang the language
     * @param themeName the key of the theme
     * @return the links as a {@link Node}.
     */
    public static NodeList getLinks(String siteName, String lang, String themeName)
    {
        return _getLinks(siteName, lang, themeName, null, false);
    }
    
    /**
     * Get the links that belong to a given theme for a given user.<br>
     * Only user access is check. The user personal links are not retrieved and none user preferences is take into account to order or hide links !
     * If IP restriction is configured, user authorization on IP is checked.
     * DO NOT CALL THIS METHOD ON A CACHEABLE PAGE !
     * @param siteName the site name
     * @param lang the language
     * @param themeName the key of the theme
     * @return the links as a {@link Node}.
     */
    public static NodeList getLinksForCurrentUser(String siteName, String lang, String themeName)
    {
        return _getLinks(siteName, lang, themeName, _currentUserProvider.getUser(), true);
    }
    
    /**
     * Determines if the user IP matches the configured internal IP range for current site
     * DO NOT CALL THIS METHOD ON A CACHEABLE PAGE !
     * @return true if the user IP is an authorized IP for internal links or if no IP restriction is configured
     */
    public static boolean isInternalIP()
    {
        String siteName = site();
        Site site = _siteManager.getSite(siteName);
        return site != null ? _directoryHelper.isInternalIP(site) : false;
    }
    
    private static NodeList _getLinks(String siteName, String lang, String themeName, UserIdentity user, boolean checkIPAuthorization)
    {
        try
        {
            Site site = _siteManager.getSite(siteName);
            
            boolean hasIPRestriction = _directoryHelper.hasIPRestriction(site);
            boolean isIPAuthorized = checkIPAuthorization ? _directoryHelper.isInternalIP(site) : false;
            
            List<DefaultLink> links = _directoryHelper.getLinks(List.of(themeName), siteName, lang);
            
            SAXTransformerFactory saxTransformerFactory = (SAXTransformerFactory) TransformerFactory.newInstance();
            TransformerHandler th = saxTransformerFactory.newTransformerHandler();
            
            DOMResult result = new DOMResult();
            th.setResult(result);
            
            th.startDocument();
            XMLUtils.startElement(th, "links");
            
            for (DefaultLink link : links)
            {
                if (_rightManager.hasReadAccess(user, link))
                {
                    _directoryHelper.saxLink(siteName, th, link, List.of(), false, hasIPRestriction, isIPAuthorized, false, false);
                }
            }
            
            XMLUtils.endElement(th, "links");
            th.endDocument();
            
            List<Node> linkNodes = new ArrayList<>();
            
            NodeList childNodes = result.getNode().getChildNodes(); 
            for (int i = 0; i < childNodes.getLength(); i++)
            {
                Node n = childNodes.item(i);
                linkNodes.add(n);
            }
            
            return new AmetysNodeList(linkNodes);
        }
        catch (Exception e) 
        {
            _logger.error("Unable to sax links directory for theme '" + themeName + " 'and site name '" + siteName + "'", e);
        }
        
        return null;
    }

}
