/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.theme;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.tag.DefaultTag;
import org.ametys.cms.tag.jcr.AbstractJCRTagProvider;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.site.SiteManager;

/**
 * Class representing a JCR themes provider for links. <br>
 */
public class JCRThemeProvider extends AbstractJCRTagProvider<DefaultTag>
{
    /** Constant for plugin node name */ 
    public static final String LINKDIRECTORY_PLUGIN_NODE_NAME = "linkdirectory";
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The theme provider extension point */
    protected ThemeProviderExtensionPoint _themeProviderEP;
    
    /** Site Manager */
    protected SiteManager _siteManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _themeProviderEP = (ThemeProviderExtensionPoint) smanager.lookup(ThemeProviderExtensionPoint.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
    }
    
    @Override
    protected void _fillCache(TraversableAmetysObject parentTagNode, DefaultTag parentTag, Map<String, DefaultTag> cache) throws RepositoryException
    {
        for (AmetysObject child : parentTagNode.getChildren())
        {
            if (child instanceof ThemeJCR)
            {
                ThemeJCR jcrTag = (ThemeJCR) child;
                DefaultTag tag = new DefaultTag(jcrTag.getId(), jcrTag.getName(), parentTag, new I18nizableText(jcrTag.getTitle()), new I18nizableText(jcrTag.getDescription()));
                cache.put(child.getName(), tag);
                
                if (parentTag != null)
                {
                    parentTag.addTag(tag);
                }
                
                _fillCache(jcrTag, tag, cache);
            }
        }
    }
    
    @Override
    protected Map<String, DefaultTag> _getCache(Map<String, Object> contextualParameters) throws RepositoryException
    {
        Request request = ContextHelper.getRequest(_context);
        if (request == null)
        {
            return new HashMap<>();
        }
        
        String siteName = (String) contextualParameters.get("siteName");
        
        @SuppressWarnings("unchecked")
        Map<String, Map<String, DefaultTag>> cache = (Map<String, Map<String, DefaultTag>>) request.getAttribute(CACHE_REQUEST_ATTRIBUTE + "$" + getId());
        if (cache == null)
        {
            cache = new HashMap<>();
            request.setAttribute(CACHE_REQUEST_ATTRIBUTE + "$" + getId(), cache);
        }
        
        Map<String, DefaultTag> siteCache = cache.get(siteName);
        if (siteCache == null)
        {
            siteCache = new HashMap<>();
            
            TraversableAmetysObject rootNode = getRootNode(contextualParameters);
            _fillCache(rootNode, null, siteCache);
            
            cache.put(siteName, siteCache);
        }
        
        return siteCache;
    }
    
    @Override
    protected ModifiableTraversableAmetysObject _getRootNode(Map<String, Object> contextualParameters, Session session) throws RepositoryException
    {
        String siteName = (String) contextualParameters.get("siteName");
        String language = (String) contextualParameters.get("language");
        @SuppressWarnings("unchecked")
        Map<String, String> user  = (Map<String, String>) contextualParameters.get("user");
        String locale = null;
        if (user != null)
        {
            locale = user.get("locale");
        }
        if (StringUtils.isBlank(siteName))
        {
            try
            {
                ModifiableTraversableAmetysObject pluginsNode = _resolver.resolveByPath("/ametys:plugins", session);
                
                ModifiableTraversableAmetysObject pluginNode = _getOrCreateNode(pluginsNode, LINKDIRECTORY_PLUGIN_NODE_NAME, "ametys:unstructured");
                ModifiableTraversableAmetysObject parentNode = pluginNode;
                
                if (StringUtils.isNotBlank(language))
                {
                    parentNode = _getOrCreateNode(pluginNode, language, "ametys:unstructured");
                }
                
                return _getOrCreateNode(parentNode, _themeProviderEP.getTagsNodeName(), _themeProviderEP.getTagsNodeType());
            }
            catch (AmetysRepositoryException e)
            {
                throw new AmetysRepositoryException("Unable to get the JCR tags root node", e);
            }
        }
        else
        {
            ModifiableTraversableAmetysObject pluginsNode = _siteManager.getSite(siteName, session).getRootPlugins();
            
            ModifiableTraversableAmetysObject pluginNode = _getOrCreateNode(pluginsNode, LINKDIRECTORY_PLUGIN_NODE_NAME, "ametys:unstructured");
            ModifiableTraversableAmetysObject parentNode = pluginNode;
            
            if (StringUtils.isNotBlank(language))
            {
                parentNode = _getOrCreateNode(pluginNode, language, "ametys:unstructured");
            }
            else if (StringUtils.isNotBlank(locale))
            {
                parentNode = _getOrCreateNode(pluginNode, locale, "ametys:unstructured");
            }
            
            return _getOrCreateNode(parentNode, _themeProviderEP.getTagsNodeName(), _themeProviderEP.getTagsNodeType());
        }
    }
    
    @Override
    public Collection<DefaultTag> getTags(String tagID, Map<String, Object> contextualParameters)
    {
        DefaultTag tag = getTag(tagID, contextualParameters);
        return tag != null ? tag.getTags().values() : null;
    }
}
