/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.theme;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.cms.tag.AbstractTagProvider;
import org.ametys.cms.tag.DefaultTag;
import org.ametys.cms.tag.Tag;
import org.ametys.plugins.linkdirectory.DirectoryHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.skin.SkinsManager;

/**
 * This class represents the themes provided by the skin
 */
public class SkinThemeProvider extends AbstractTagProvider<DefaultTag> implements Serviceable
{
    /** The source resolver */
    protected SourceResolver _resolver;
    
    /** The tags */
    protected Map<String, List<String>> _skinLocalIds;
    /** The tags */
    protected Map<String, Map<String, DefaultTag>> _skinTags;
    
    private SiteManager _siteManager;
    private SkinsManager _skinsManager;

    private DirectoryHelper _directoryHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _skinsManager = (SkinsManager) smanager.lookup(SkinsManager.ROLE);
        _directoryHelper = (DirectoryHelper) smanager.lookup(DirectoryHelper.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _skinLocalIds = new HashMap<>();
        
        _id = configuration.getAttribute("id");
        _label = configureLabel(configuration, "plugin." + _pluginName);
        _description = configureDescription(configuration, "plugin." + _pluginName);
        
        if (_skinTags == null)
        {
            _skinTags = new HashMap<>();
        }
        
        for (String skinName : _skinsManager.getSkins())
        {
            try
            {
                initializeTags(skinName);
            }
            catch (Exception e)
            {
                throw new ConfigurationException("Unable to load tags configuration values for skin " + skinName, e);
            }
        }
    }
    
    @Override
    public boolean hasTag(String tagID, Map<String, Object> contextualParameters)
    {
        return getTag(tagID, contextualParameters) != null;
    }
    
    @Override
    public DefaultTag getTag(String tagID, Map<String, Object> contextualParameters)
    {
        Map<String, DefaultTag> tags = getTags(contextualParameters);
        return tags != null ? _recursiveSearchTags(tags, tagID) : null;
    }

    @Override
    public Collection<DefaultTag> getTags(String tagID, Map<String, Object> contextualParameters)
    {
        DefaultTag tag = getTag(tagID, contextualParameters);
        return tag != null ? tag.getTags().values() : null;
    }
    
    private DefaultTag _recursiveSearchTags(Map<String, DefaultTag> tags, String tagID)
    {
        if (tags.containsKey(tagID))
        {
            return tags.get(tagID);
        }
        
        for (DefaultTag child : tags.values())
        {
            DefaultTag tag = _recursiveSearchTags(child.getTags(), tagID);
            if (tag != null)
            {
                return tag;
            }
        }
        
        return null;
    }
    
    @Override
    public Map<String, DefaultTag> getTags(Map<String, Object> contextualParameters)
    {
        if (contextualParameters.get("siteName") == null)
        {
            // contextualParameters#containsKey not sufficient here because the in some case the siteName key can be set to null
            return null;
        }
        
        Site site = _siteManager.getSite((String) contextualParameters.get("siteName"));
        
        if (site == null)
        {
            String errorMessage = "Unable to load tags configuration values for site " + (String) contextualParameters.get("siteName");
            getLogger().error(errorMessage);
            return null;
        }
        
        String skin = site.getSkinId();
        
        if (!_skinTags.containsKey(skin))
        {
            try
            {
                initializeTags(skin);
            }
            catch (Exception e)
            {
                String errorMessage = "Unable to load tags configuration values for skin " + skin;
                getLogger().error(errorMessage, e);
            }
        }
        
        return _skinTags.get(skin);
    }
    
    /**
     * Initialize a skin's themes from the themes file.
     * @param skinName the name of the skin to initialize themes.
     * @throws Exception if an error occurs.
     */
    protected void initializeTags(String skinName) throws Exception
    {
        // Obsolete location
        Source otherSrc = null;
        try
        {
            otherSrc = _resolver.resolveURI("skin:" + skinName + "://conf/link-themes.xml");
            if (otherSrc.exists())
            {
                getLogger().error("In skin '" + skinName + "' (or one of its parent skin) the 'conf/link-themes.xml' file location is obsolete AND NOT SUPPORTED ANYMORE. Move the file conf/link-themes.xml to conf/link-directory.xml");
            }
        }
        finally 
        {
            _resolver.release(otherSrc);
        }
        
        
        // Right inheritable location
        Configuration configuration = _directoryHelper.getSkinLinksConfiguration(skinName);
        
        if (!_skinLocalIds.containsKey(skinName))
        {
            _skinLocalIds.put(skinName, new ArrayList<>());
        }
        
        Map<String, DefaultTag> tags = configureTags(configuration, skinName, null, "skin." + skinName);
        _skinTags.put(skinName, tags);
    }
    
    /**
     * Configure themes from the passed configuration
     * @param configuration The configuration
     * @param skinName the skin name
     * @param parent The parent theme if any (as recursion is disabled, should be null)
     * @param defaultCatalogue The default catalog for i18n
     * @return a Set of {@link DefaultTag}
     * @throws ConfigurationException if configuration is invalid
     */
    protected Map<String, DefaultTag> configureTags (Configuration configuration, String skinName, DefaultTag parent, String defaultCatalogue)  throws ConfigurationException
    {
        Map<String, DefaultTag> themes = new HashMap<>();
        
        Configuration[] tagsConfiguration = configuration.getChild("definitions").getChildren("theme");
        for (Configuration tagConfiguration : tagsConfiguration)
        {
            String id = tagConfiguration.getAttribute("id", null);
            if (id == null)
            {
                getLogger().error("Missing attributed named \"id\" for theme configuration at " + tagConfiguration.getLocation() + ". Theme is ignored.");
            }
            else if (!Tag.NAME_PATTERN.matcher(id).matches())
            {
                getLogger().error("Invalid tag ID \"" + id + "\" for theme configuration at " + tagConfiguration.getLocation() + ". It must match the pattern " + Tag.NAME_PATTERN.toString() + ". Theme is ignored.");
            }
            else if (_skinLocalIds.get(skinName).contains(id))
            {
                getLogger().error("A tag with the ID \"" + id + "\" for theme configuration at " + tagConfiguration.getLocation() + " already exists. Theme is ignored.");
            }
            else
            {
                _skinLocalIds.get(skinName).add(id);
                
                I18nizableText label = configureLabel (tagConfiguration, defaultCatalogue);
                I18nizableText description = configureDescription (tagConfiguration, defaultCatalogue);
                DefaultTag theme = new DefaultTag(id, id, parent, label, description);
                
                // No recursivity for themes, if it become a thing, uncomment this :
//                Map<String, DefaultTag> childTags = configureTags(tagConfiguration, skinName, theme, defaultCatalogue);
//                theme.setTags(childTags);
                themes.put(id, theme);
            }
            
        }
        
        return themes;
    }
}
