/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.linkdirectory.LinkDirectoryThemesInputDataHelper.ConfiguredThemesInputData;
import org.ametys.plugins.linkdirectory.LinkDirectoryThemesInputDataHelper.ThemesInputData;
import org.ametys.plugins.linkdirectory.repository.DefaultLink;
import org.ametys.web.inputdata.InputData;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

/**
 * Input data for the link directory user preferences in thumbnails mode 
 */
public class LinkDirectoryInputData extends AbstractLogEnabled implements Contextualizable, InputData, Serviceable
{
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The Avalon context */
    private Context _context;
    
    private DirectoryHelper _directoryHelper;
    
    /** The link directory themes input data helper */
    protected LinkDirectoryThemesInputDataHelper _linkDirectoryThemesInputDataHelper;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context; 
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _linkDirectoryThemesInputDataHelper = (LinkDirectoryThemesInputDataHelper) manager.lookup(LinkDirectoryThemesInputDataHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _directoryHelper = (DirectoryHelper) manager.lookup(DirectoryHelper.ROLE);
    }
    
    @Override
    public boolean isCacheable(Site site, Page currentPage)
    {
        Request request = ContextHelper.getRequest(_context);
        
        String template = _getTemplate(request);
        if (template == null)
        {
            return true;
        }
        
        try
        {
            ConfiguredThemesInputData configuredThemesInputData = _linkDirectoryThemesInputDataHelper.getThemesInputData(site.getSkinId());
            List<ThemesInputData> themesInputData = _getThemesForSkinAndTemplate(configuredThemesInputData, template);
            if (themesInputData.isEmpty())
            {
                // The current template is not configured for a link directory input data
                return true;
            }
            
            for (ThemesInputData themeInputData : themesInputData)
            {
                if (themeInputData.configurable() || themeInputData.displayUserLinks())
                {
                    // The applications are configurable
                    return false;
                }
            }
            
            // Find the configured theme ids for this template
            String language = _directoryHelper.getLanguage(request);
            List<String> configuredThemesNames = themesInputData.stream()
                .map(inputData -> _getConfiguredThemes(inputData, language))
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
            String siteName = _directoryHelper.getSiteName(request);
            
            return !_directoryHelper.hasRestrictions(siteName, language, configuredThemesNames) && !_directoryHelper.hasInternalUrl(siteName, language, configuredThemesNames);
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred while retrieving information from the skin configuration", e);
            // Configuration file is not readable => toSAX method will not generate any xml
            return true;
        }
    }

    @Override
    public void toSAX(ContentHandler contentHandler) throws ProcessingException
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Get the current user's login if he is in the front office
        UserIdentity user = _currentUserProvider.getUser();

        String template = _getTemplate(request);
        if (template == null)
        {
            getLogger().info("There is no current template");
            return; 
        }
        
        String skinId = _getSkin(request);
        if (skinId == null)
        {
            getLogger().info("There is no current skin");
            return; 
        }
        
        try
        {
            contentHandler.startDocument();

            ConfiguredThemesInputData configuredThemesInputData = _linkDirectoryThemesInputDataHelper.getThemesInputData(skinId);
            List<ThemesInputData> themesInputDatas = _getThemesForSkinAndTemplate(configuredThemesInputData, template);
            
            // Is there an error in the configuration file ?
            String error = configuredThemesInputData.error();
            if (StringUtils.isNotBlank(error))
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("error", error);
                XMLUtils.createElement(contentHandler, "linkDirectory", attrs);
            }
            else
            {
                String language = _directoryHelper.getLanguage(request);
                String siteName = _directoryHelper.getSiteName(request);
                for (ThemesInputData themesInputData : themesInputDatas)
                {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("applicable", Boolean.TRUE.toString());
                    attrs.addCDATAAttribute("configurable", String.valueOf(themesInputData.configurable()));
                    attrs.addCDATAAttribute("displayUserLinks", String.valueOf(themesInputData.displayUserLinks()));
                    attrs.addCDATAAttribute("id", themesInputData.id());
                    
                    XMLUtils.startElement(contentHandler, "linkDirectory", attrs);
                    
                    List<String> configuredThemesNames = _getConfiguredThemes(themesInputData, language);
                    if (configuredThemesNames != null)
                    {
                        Map<String, List<String>> themesMap = _directoryHelper.getThemesMap(configuredThemesNames, siteName, language);
                        List<String> correctThemesIds = themesMap.get("themes");
                        List<String> unknownThemesNames = themesMap.get("unknown-themes");

                        _saxThemes(contentHandler, correctThemesIds, unknownThemesNames);
                        _saxLinks(contentHandler, user, request, correctThemesIds, themesInputData.displayUserLinks(), themesInputData.configurable(), themesInputData.id());
                    }
                    
                    XMLUtils.endElement(contentHandler, "linkDirectory");
                }
            }
        }
        catch (Exception e)
        {
            getLogger().error("An exception occurred during the processing of the link directory's input data" , e);
        }
    }
    
    private void _saxThemes(ContentHandler contentHandler, List<String> themeIds, List<String> unknownThemesNames) throws SAXException
    {
        if (!themeIds.isEmpty())
        {
            XMLUtils.startElement(contentHandler, "themes");
            for (String themeId : themeIds)
            {
                XMLUtils.createElement(contentHandler, "theme", themeId);
            }
            XMLUtils.endElement(contentHandler, "themes");
        }
        
        if (!unknownThemesNames.isEmpty())
        {
            AttributesImpl attr = new AttributesImpl();
            attr.addCDATAAttribute("count", Integer.toString(unknownThemesNames.size()));
            XMLUtils.createElement(contentHandler, "unknown-themes", attr, StringUtils.join(unknownThemesNames, ", "));
        }
    }

    private void _saxLinks(ContentHandler contentHandler, UserIdentity user, Request request, List<String> themeIds, boolean displayUserLinks, boolean configurable, String specificContext) throws ProcessingException
    {
        String language = _directoryHelper.getLanguage(request);
        String siteName = _directoryHelper.getSiteName(request);
        try
        {
            // SAX common links
            List<DefaultLink> links = _directoryHelper.getLinks(themeIds, siteName, language);
            
            List<DefaultLink> userLinks = null;
            if (user != null && displayUserLinks)
            {
                userLinks = _directoryHelper.getUserLinks(siteName, language, user).stream().collect(Collectors.toList());
            }
            
            
            // SAX the user own links
            XMLUtils.startElement(contentHandler, "links");
            
            try
            {
                String storageContext = siteName + "/" + language;
                if (StringUtils.isNotEmpty(specificContext))
                {
                    storageContext += "/" + specificContext;
                }
                _directoryHelper.saxLinks(siteName, contentHandler, links, userLinks, themeIds, configurable, _directoryHelper.getContextVars(request), storageContext, user);
            }
            catch (Exception e)
            {
                getLogger().error("An exception occurred while saxing the links", e);
            }
            
            XMLUtils.endElement(contentHandler, "links");
        }
        catch (Exception e)
        {
            throw new ProcessingException("An error occurred while retrieving or saxing the links", e);
        }
    }

    /**
     * Retrieve the configured themes names defined in the skin file link-directory.xml for the current input data and the current language
     * @param themesInputData Can be an empty {@link String}
     * @param lang language to filter by. Themes with lang=null will always be returned.
     * @return the list of configured themes ids, can be empty, cannot be null
     */
    private List<String> _getConfiguredThemes(ThemesInputData themesInputData, String lang)
    {
        return themesInputData.themes()
            .stream()
            .filter(t -> t.get("lang") == null || t.get("lang").equals(lang))
            .map(t -> t.get("id"))
            .collect(Collectors.toList());
    }
    
    private List<ThemesInputData> _getThemesForSkinAndTemplate(ConfiguredThemesInputData configuredThemesInputData, String template) 
    {
        List<ThemesInputData> themesInputDatas = configuredThemesInputData.themesInputDatas();
        if (themesInputDatas == null)
        {
            return List.of();
        }
        
        return themesInputDatas.stream()
            .filter(t -> _filterByTemplate(t, template))
            .collect(Collectors.toList());
    }
    
    private boolean _filterByTemplate(ThemesInputData theme, String template)
    {
        List<String> templates = theme.templates();
        return templates.contains(template) || templates.contains(LinkDirectoryThemesInputDataHelper.WILDCARD); 
    }
       
    /**
     * Get the current template 
     * @param request the request
     * @return the current template
     */
    private String _getTemplate(Request request)
    {
        return (String) request.getAttribute("template");
    }
    
     /**
      * Get the current skin 
      * @param request the request
      * @return the current skin
      */
    private String _getSkin(Request request)
    {
        return (String) request.getAttribute("skin");
    }
}
