/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Helper for input data for the link directory user preferences in thumbnails mode 
 */
public class LinkDirectoryThemesInputDataHelper extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** The component role */
    public static final String ROLE = LinkDirectoryThemesInputDataHelper.class.getName();

    /** The wildcard */
    public static final String WILDCARD = "*";
    
    private static final String __CONF_FILE_PATH = "skin://conf/link-directory.xml";
    private static final String __THEMES_CACHE = LinkDirectoryThemesInputDataHelper.class.getName() + "$skinInputDataThemesCache";
    
    private SourceResolver _sourceResolver;
    private DirectoryHelper _directoryHelper;
    private AbstractCacheManager _cacheManager;
    private Map<String, String> _configurationError;
    
    /** The last time the file was loaded */
    private Map<String, Long> _lastConfUpdate;

    @Override
    public void initialize() throws Exception
    {
        _lastConfUpdate = new HashMap<>();
        _configurationError = new HashMap<>();
        _cacheManager.createMemoryCache(__THEMES_CACHE,
                new I18nizableText("plugin.link-directory", "PLUGINS_LINK_DIRECTORY_CACHE_THEMES_LABEL"),
                new I18nizableText("plugin.link-directory", "PLUGINS_LINK_DIRECTORY_CACHE_THEMES_DESCRIPTION"),
                true,
                null);
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _directoryHelper = (DirectoryHelper) manager.lookup(DirectoryHelper.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
    }
    
    /**
     * Get the list of the themes input data for a given skin
     * @param skinId the given skin id
     * @return the list of the themes input data
     */
    public ConfiguredThemesInputData getThemesInputData(String skinId)
    {
        try
        {
            _updateConfigurationValues(skinId);
            if (_configurationError.containsKey(skinId))
            {
                // Some configuration errors occurred, return empty list
                return new ConfiguredThemesInputData(List.of(), _configurationError.get(skinId));
            }
            
            return new ConfiguredThemesInputData(_getThemesCache().get(skinId, k -> new ArrayList<>()), null);
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred while retrieving information from the skin configuration", e);
            // Configuration file is not readable => toSAX method will not generate any xml
            return new ConfiguredThemesInputData(List.of(), null);
        }
    }

    /**
     * Update the configuration values : read them if the map is empty, update them if the file was changed or simply return them
     * @param skinId The skin
     * @throws Exception if an exception occurs
     */
    private void _updateConfigurationValues(String skinId) throws Exception
    {
        Source source = null;
        try
        {
            source = _sourceResolver.resolveURI(__CONF_FILE_PATH);
            if (source.exists())
            {
                _cacheConfigurationValues(source, skinId, !_getThemesCache().hasKey(skinId));
            }
            else
            {
                if (getLogger().isInfoEnabled())
                {
                    getLogger().info("There is no configuration file at path '" + __CONF_FILE_PATH + "' (no input data for link directory).");
                }
                
                _lastConfUpdate.put(skinId, (long) 0);
                _getThemesCache().put(skinId, null);
            }
        }
        finally
        {
            if (_sourceResolver != null && source != null)
            {
                _sourceResolver.release(source);
            }
        }
    }
    
    /**
     * Read the configuration values and store them
     * @param source the file's source
     * @param skinId The skin
     * @param forceRead true to force reload of values even if the file was not modified
     */
    private synchronized void _cacheConfigurationValues (Source source, String skinId, boolean forceRead)
    {
        long lastModified = source.getLastModified();
        if (!forceRead && _lastConfUpdate.containsKey(skinId) && _lastConfUpdate.get(skinId) != 0 && lastModified == _lastConfUpdate.get(skinId))
        {
            // While waiting for synchronized, someone else may have updated the cache
            return;
        }

        List<ThemesInputData> themesCache = new ArrayList<>();

        getLogger().info("Caching configuration");
        
        try
        {
            Configuration configuration = _directoryHelper.getSkinLinksConfiguration(skinId);
            
            Configuration[] themesConfigurations = configuration.getChild("inputdata").getChildren("themes");
            
            for (Configuration themesConfiguration : themesConfigurations)
            {
                List<Map<String, String>> themes = new ArrayList<> ();
                
                Configuration[] themeConfigurations = themesConfiguration.getChildren();
                for (Configuration themeConfiguration : themeConfigurations)
                {
                    Map<String, String> theme = new HashMap<> ();
                    String id = themeConfiguration.getAttribute("id", null);
                    theme.put("id", id);
                    theme.put("lang", themeConfiguration.getAttribute("lang", null));
                    themes.add(theme);
                }
                
                String[] templates = StringUtils.split(themesConfiguration.getAttribute("templates", WILDCARD), ',');
                
                ThemesInputData themeInputData = new ThemesInputData(themesConfiguration.getAttribute("inputDataId", StringUtils.EMPTY), Arrays.asList(templates), themes, themesConfiguration.getAttributeAsBoolean("configurable", false), themesConfiguration.getAttributeAsBoolean("displayUserLinks", false));
                themesCache.add(themeInputData);
            }
            
            _configurationError.remove(skinId);
            _getThemesCache().put(skinId, themesCache);
            _lastConfUpdate.put(skinId, source.getLastModified());
        }
        catch (Exception e)
        {
            getLogger().warn("An error occured while getting the configuration's file values", e);
            _configurationError.put(skinId, e.getMessage());
        }
    }
       
    private Cache<String, List<ThemesInputData>> _getThemesCache()
    {
        return _cacheManager.get(__THEMES_CACHE);
    }
    
    /**
     * A record representing a themes input data
     * @param id the input data id
     * @param templates the templates of the themes
     * @param themes the themes in the input data
     * @param configurable <code>true</code> if the themes are configurable
     * @param displayUserLinks <code>true</code> to have the user links
     */
    public record ThemesInputData(String id, List<String> templates, List<Map<String, String>> themes, boolean configurable, boolean displayUserLinks) { /* */ }
    
    /**
     * A record representing the list of themes input datas with the configuration errors
     * @param themesInputDatas the themes input data
     * @param error the error
     */
    public record ConfiguredThemesInputData(List<ThemesInputData> themesInputDatas, String error) { /* */ }
}
