/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory;

import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.data.Binary;
import org.ametys.cms.transformation.AbstractURIResolver;
import org.ametys.cms.transformation.ConsistencyChecker.CHECK;
import org.ametys.cms.transformation.URIResolver;
import org.ametys.core.util.FilenameUtils;
import org.ametys.core.util.ImageResolverHelper;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.linkdirectory.repository.DefaultLink;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.URIPrefixHandler;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * {@link URIResolver} for type "link-data".<br>
 * These links or images point to a file from the data of a {@link Link}.
 */
public class LinkMetadataURIResolver extends AbstractURIResolver implements Serviceable, Contextualizable, PluginAware
{
    private static final Pattern _OBJECT_URI_PATTERN = Pattern.compile("([^?]*)\\?objectId=(.*)");
    
    private AmetysObjectResolver _resolver;
    private URIPrefixHandler _prefixHandler;
    private RenderingContextHandler _renderingContexthandler;
    private SiteManager _siteManager;
    
    /** The context */
    private Context _context;

    private String _pluginName;
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _prefixHandler = (URIPrefixHandler) manager.lookup(URIPrefixHandler.ROLE);
        _renderingContexthandler = (RenderingContextHandler) manager.lookup(RenderingContextHandler.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }
    
    @Override
    public String getType()
    {
        return "link-data";
    }
    
    @Override
    protected String _resolve(String uri, String uriArgument, boolean download, boolean absolute, boolean internal)
    {
        try
        {
            Request request = ContextHelper.getRequest(_context);
            
            Info info = _getInfo(uri, request);
            
            DefaultLink link = info.getLink();
            String dataPath = info.getDataPath();
            
            if (link == null)
            {
                throw new IllegalStateException("Cannot resolve a local link to an unknown link for uri " + request.getRequestURI());
            }
            
            Binary binary = link.getValue(dataPath);
            
            String filename = FilenameUtils.encodeName(binary.getFilename());
            
            String baseName = org.apache.commons.io.FilenameUtils.getBaseName(filename);
            String extension = org.apache.commons.io.FilenameUtils.getExtension(filename);
            
            StringBuilder resultPath = new StringBuilder();
            
            resultPath.append("/_plugins/")
                      .append(_pluginName)
                      .append("/_links")
                      .append(FilenameUtils.encodePath(link.getPath()))
                      .append("/_data/")
                      .append(dataPath)
                      .append("/")
                      .append(baseName)
                      .append(uriArgument)
                      .append(extension.isEmpty() ? "" : "." + extension);
              
            String result = _getUri(resultPath.toString(), link, absolute, internal);
              
            Map<String, String> params = new HashMap<>();
            params.put("objectId", link.getId());
            if (download)
            {
                params.put("download", "true");
            }
            
            return URIUtils.encodeURI(result, params);
        }
        catch (Exception e)
        {
            throw new IllegalStateException(e);
        }
    }

    @Override
    protected String resolveImageAsBase64(String uri, int height, int width, int maxHeight, int maxWidth, int cropHeight, int cropWidth)
    {
        try
        {
            Request request = ContextHelper.getRequest(_context);
            
            Info info = _getInfo(uri, request);
            
            DefaultLink link = info.getLink();
            String dataPath = info.getDataPath();
            
            if (link == null)
            {
                throw new IllegalStateException("Cannot resolve a local link to an unknown link for uri " + request.getRequestURI());
            }
            
            Binary binary = link.getValue(dataPath);
            
            try (InputStream dataIs = binary.getInputStream())
            {
                return ImageResolverHelper.resolveImageAsBase64(dataIs, binary.getMimeType(), height, width, maxHeight, maxWidth, cropHeight, cropWidth);
            }
        }
        catch (Exception e)
        {
            throw new IllegalStateException(e);
        }
    }
    
    public String getMimeType(String uri)
    {
        try
        {
            Request request = ContextHelper.getRequest(_context);
            
            Info info = _getInfo(uri, request);
            
            DefaultLink link = info.getLink();
            String dataPath = info.getDataPath();
            
            if (link == null)
            {
                throw new IllegalStateException("Cannot resolve a local link to an unknown link for uri " + request.getRequestURI());
            }
            
            Binary binary = link.getValue(dataPath);
            
            return binary.getMimeType();
        }
        catch (Exception e)
        {
            throw new IllegalStateException(e);
        }
    }
    
    private String _getUri(String path, DefaultLink object, boolean absolute, boolean internal)
    {
        String siteName = object.getSiteName();
        
        if (internal)
        {
            return "cocoon://" + siteName + path;
        }
        else if (absolute)
        {
            if (_renderingContexthandler.getRenderingContext() == RenderingContext.FRONT)
            {
                Site site = _siteManager.getSite(siteName);
                
                String[] aliases = site.getUrlAliases();
                return aliases[Math.abs(path.hashCode()) % aliases.length] + path;
            }
            
            return _prefixHandler.getAbsoluteUriPrefix() + "/" + siteName + path; 
        }
        else
        {
            return _prefixHandler.getUriPrefix(siteName) + path;
        }
    }
    
    @Override
    public CHECK checkLink(String uri, boolean shortTest)
    {
        return CHECK.SUCCESS;
    }
    
    @Override
    public I18nizableText getLabel(String uri)
    {
        try
        {
            Request request = ContextHelper.getRequest(_context);
            Info info = _getInfo(uri, request);
            return new I18nizableText("plugin.cms", "PLUGINS_CMS_LINK_METADATA_LABEL", Collections.singletonList(info.getDataPath()));
        }
        catch (UnknownAmetysObjectException e)
        {
            return new I18nizableText("plugin.cms", "PLUGINS_CMS_LINK_METADATA_UNKNOWN");
        }
    }
    
    private Info _getInfo(String uri, Request request)
    {
        Info info = new Info();
        
        Matcher matcher = _OBJECT_URI_PATTERN.matcher(uri);
        
        // Test if the URI contains an object ID.
        if (matcher.matches())
        {
            info.setDataPath(matcher.group(1));
            String objectId = matcher.group(2);
            
            DefaultLink link = _resolver.resolveById(objectId);
            info.setLink(link);
        }
        else
        {
            throw new IllegalStateException("Missing objectId parameter to resolve a local link for uri " + request.getRequestURI());
        }
        
        return info;
    }

    /**
     * data information.
     */
    protected class Info
    {
        private String _dataPath;
        private DefaultLink _link;
        
        /**
         * Get the data path.
         * @return the data path.
         */
        public String getDataPath()
        {
            return _dataPath;
        }
        
        /**
         * Set the data path.
         * @param path the data path to set
         */
        public void setDataPath(String path)
        {
            _dataPath = path;
        }
        
        /**
         * Get the link.
         * @return the link
         */
        public DefaultLink getLink()
        {
            return _link;
        }
        
        /**
         * Set the link.
         * @param link the link to set
         */
        public void setLink(DefaultLink link)
        {
            _link = link;
        }
    }
}
