/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.dynamic;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.plugins.linkdirectory.DirectoryHelper;
import org.ametys.plugins.linkdirectory.Link.LinkStatus;
import org.ametys.plugins.linkdirectory.repository.DefaultLink;

/**
 * This generator sax the dynamic information provided by {@link DynamicInformationProvider}s
 */
public class DynamicInformationGenerator extends ServiceableGenerator
{
    private DirectoryHelper _directoryHelper;
    private DynamicInformationProviderExtensionPoint _dynamicInfoExtensionPoint;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _directoryHelper = (DirectoryHelper) serviceManager.lookup(DirectoryHelper.ROLE);
        _dynamicInfoExtensionPoint = (DynamicInformationProviderExtensionPoint) serviceManager.lookup(DynamicInformationProviderExtensionPoint.ROLE);
    }

    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        int maxItems = Integer.valueOf(request.getParameter("maxItems"));
        
        String[] themeIds = request.getParameter("themes") != null ? request.getParameter("themes").split(",") : new String[0];
        String siteName = request.getParameter("site");
        String lang = request.getParameter("lang");
        
        List<DefaultLink> links = _directoryHelper.getLinks(Arrays.asList(themeIds), siteName, lang);

        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "dynamic-information");
        for (DefaultLink defaultLink : links)
        {
            if (defaultLink.getStatus() != LinkStatus.BROKEN)
            {
                String extensionId = defaultLink.getDynamicInformationProvider();
                if (StringUtils.isNotBlank(extensionId))
                {
                    DynamicInformationProvider provider = _dynamicInfoExtensionPoint.getExtension(extensionId);
                    if (provider != null)
                    {
                        String linkId = defaultLink.getId();
                        try
                        {
                            provider.saxDynamicInformation(contentHandler, linkId, siteName, lang, maxItems);
                        }
                        catch (Exception e)
                        {
                            getLogger().error("Unable to sax dynamic information from provider of id '" + provider.getId() + "'", e);
                        }
                    }
                }
            }
        }
        XMLUtils.endElement(contentHandler, "dynamic-information");
        contentHandler.endDocument();
    }
}
