/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.dynamic;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.net.URL;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.XMLConstants;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.compress.utils.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.XMLReader;

import org.ametys.core.util.URIUtils;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;


/**
 * Implementation of {@link DynamicInformationProvider} allowing to SAX information from a confiured URL.
 */
public class URLBasedDynamicInformationProvider extends AbstractLogEnabled implements DynamicInformationProvider, Configurable, PluginAware, Serviceable
{
    private static final Pattern __CONFIG_PARAM_PATTERN = Pattern.compile("\\$\\{config:([^}]+)\\}");
    private static final Pattern __SITE_CONFIG_PARAM_PATTERN = Pattern.compile("\\$\\{site-config:([^}]+)\\}");
    
    private String _url;
    
    private SiteManager _siteManager;

    private String _pluginName;

    private I18nizableText _label;

    private String _id;

    private SourceResolver _sourceResolver;

    public void service(ServiceManager manager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }

    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
        _id = id;
    }

    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _url = configuration.getChild("url").getValue();
        _label = I18nizableText.parseI18nizableText(configuration.getChild("label"), "plugin." + _pluginName, "");
    }

    @Override
    public void saxDynamicInformation(ContentHandler contentHandler, String linkId, String siteName, String lang, int nbItems) throws Exception
    {
        String url = _handleVariable(_url, siteName, lang, nbItems);
        
        try (InputStream is = _sourceResolver.resolveURI(url).getInputStream())
        {
            byte[] asByteArray = IOUtils.toByteArray(is);
            
            if (_xsdValidation(new ByteArrayInputStream(asByteArray)))
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("id", linkId);
                XMLUtils.startElement(contentHandler, "dynamic-information", attrs);
                _saxInformation(contentHandler, new ByteArrayInputStream(asByteArray));
                XMLUtils.endElement(contentHandler, "dynamic-information");
            }
        }
    }
    
    private String _handleVariable(String url, String siteName, String lang, int nbItems)
    {
        int startParamIndex = url.indexOf("?");
        
        String convertUrl = url;
        convertUrl = StringUtils.replace(convertUrl, "${lang}", lang);
        convertUrl = StringUtils.replace(convertUrl, "${site}", siteName);
        convertUrl = StringUtils.replace(convertUrl, "${maxItems}", String.valueOf(nbItems));
        
        // Resolve configuration parameters
        Matcher confMatcher = __CONFIG_PARAM_PATTERN.matcher(convertUrl);
        while (confMatcher.find())
        {
            String paramName = confMatcher.group(1);
            String paramValue = Config.getInstance().getValue(paramName);
            
            String encodedValue = confMatcher.start() > startParamIndex ? URIUtils.encodeParameter(paramValue) : URIUtils.encodePath(paramValue);
            convertUrl = StringUtils.replace(convertUrl, confMatcher.group(0), encodedValue);
        }
        
        // Resolve site configuration parameters
        Matcher siteConfMatcher = __SITE_CONFIG_PARAM_PATTERN.matcher(convertUrl);
        Site site = _siteManager.getSite(siteName);
        while (siteConfMatcher.find())
        {
            String paramName = confMatcher.group(1);
            String paramValue = site.getValue(paramName);
            
            String encodedValue = siteConfMatcher.start() > startParamIndex ? URIUtils.encodeParameter(paramValue) : URIUtils.encodePath(paramValue);
            convertUrl = StringUtils.replace(convertUrl, siteConfMatcher.group(0), encodedValue);
        }
        
        return convertUrl;
    }

    /**
     * Verify if an XML document is an instance of a specified XML schema.
     * 
     * @param xml The XML feed.
     * @return True if XML verify the xml schema and false if not.
     */
    private boolean _xsdValidation(InputStream xml)
    {
        try
        {
            SchemaFactory factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
            URL schemaURL = getClass().getResource("dynamic-information.xsd");
            Schema schema = factory.newSchema(schemaURL);
            Validator validator = schema.newValidator();
            validator.validate(new StreamSource(xml));
            return true;
        }
        catch (Exception e)
        {
            getLogger().error("Unable to validate XSD for '{}' ({}) information at '{}' url.", _label, _id, _url, e);
            return false;
        }
    }

    /**
     * Sax xml feed returned by URL provider.
     * @param contentHandler The content handler to sax into
     * @param xml The xml feed.
     * @throws Exception if exception error occurs.
     */
    private void _saxInformation(ContentHandler contentHandler, InputStream xml) throws Exception
    {
        SAXParserFactory saxFactory = SAXParserFactory.newInstance();
        saxFactory.setNamespaceAware(true);
        XMLReader reader = saxFactory.newSAXParser().getXMLReader();
        DynamicInformationHandler dynamicInfoHandler = new DynamicInformationHandler(contentHandler);
        reader.setContentHandler(dynamicInfoHandler);
        reader.parse(new InputSource(xml));
    }

    /**
     * Get the url of the WebService
     * @return url of the WebService
     */
    public String getUrl()
    {
        return _url;
    }

    @Override
    public String getId()
    {
        return _id;
    }

    @Override
    public I18nizableText getLabel()
    {
        return _label;
    }

}
