/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.link;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.upload.Upload;
import org.ametys.core.upload.UploadManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.linkdirectory.DirectoryHelper;
import org.ametys.plugins.linkdirectory.Link;
import org.ametys.plugins.linkdirectory.Link.LinkType;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.url.UrlPreviewComponent;

/**
 * Adds a new user link in Ametys.
 */
public class AddUserLinkAction extends ServiceableAction
{
    private static int _ICO_REQUEST_TIMEOUT = 3000;
    
    private static final List<String> __ICO_SUPPORTED_MIMETYPES = List.of("image/jpeg", "image/jpg", "image/png", "image/svg+xml");
    private static final List<String> __ICO_SUPPORTED_EXTENSIONS = List.of(".jpeg", ".jpg", ".png", ".svg");
    
    /** The DAO for {@link Link}s */
    protected LinkDAO _linkDAO;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The directory helper */
    protected DirectoryHelper _directoryHelper;
    /** The site manager */
    protected SiteManager _siteManager;
    /** The url preview component */
    protected UrlPreviewComponent _urlPreviewComponent;
    /** The upload manager */
    protected UploadManager _uploadManager;
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _linkDAO = (LinkDAO) smanager.lookup(LinkDAO.ROLE);
        _directoryHelper = (DirectoryHelper) smanager.lookup(DirectoryHelper.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _urlPreviewComponent = (UrlPreviewComponent) smanager.lookup(UrlPreviewComponent.ROLE);
        _uploadManager = (UploadManager) smanager.lookup(UploadManager.ROLE);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Map<String, Object> result = new HashMap<>();
        
        UserIdentity user = _currentUserProvider.getUser();
        
        // Check user authenticated
        if (user != null)
        {
            // Create link
            Map<String, Object> createParameters = new HashMap<>();
            String siteName = request.getParameter("siteName");
            String lang = request.getParameter("lang");
            String url = request.getParameter("url");
            //The color key, default color will be used if it is not present or empty
            String color = request.getParameter("color");
            
            createParameters.put("siteName", siteName);
            createParameters.put("lang", lang);
            createParameters.put("url-type", LinkType.URL.toString());
            createParameters.put("url", url);
            createParameters.put("title", request.getParameter("title"));
            createParameters.put("color", color);
            
            String userAgent = request.getHeader("User-Agent");
            Upload upload = _getLinkFavicon(url, user, userAgent);
            if (upload != null)
            {
                Map<String, Object> linkMap = new HashMap<>();
                linkMap.put("id", upload.getId());
                linkMap.put("type", "external");
                createParameters.put("picture", _jsonUtils.convertObjectToJson(linkMap));
            }
            
            _handleTheme(request, createParameters, siteName, lang);
            
            Map<String, Object> createdLinkresult = _linkDAO.createUserLink(createParameters);
            
            if (createdLinkresult.containsKey("already-exists"))
            {
                result.put("error", "already-exists");
            }
            else
            {
                result.put("error", "none");
                result.putAll(createdLinkresult);
            }
        }
        else
        {
            result.put("error", "unauthenticated-user");
        }
        
        Response response = ObjectModelHelper.getResponse(objectModel);
        response.setHeader("Access-Control-Allow-Origin", request.getHeader("Origin"));
        response.setHeader("Access-Control-Allow-Credentials", "true");
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    private void _handleTheme(Request request, Map<String, Object> createParameters, String siteName, String lang)
    {
        String themeName = request.getParameter("themes");
        
        List<String> themes = new ArrayList<>();
        // We have a theme name ...
        if (StringUtils.isNotBlank(themeName))
        {
            // ... retrieve JCR id from theme name
            if (_directoryHelper.themeExists(themeName, siteName, lang))
            {
                themes.add(themeName);
            }
        }
        // if not, just send an empty list for the user link creation
        createParameters.put("themes", themes);
    }
    
    /**
     * Set the favicon of the link url
     * @param url the url of the favicon
     * @param user the user
     * @param userAgent The user agent
     * @return the favicon upload map
     */
    protected Upload _getLinkFavicon(String url, UserIdentity user, String userAgent)
    {
        Upload upload = null;
        try
        {
            String favIconLink = _urlPreviewComponent.getFavicon(url);
            
            if (StringUtils.isBlank(favIconLink))
            {
                return null;
            }
            
            URL favIconURL = new URL(favIconLink);
            String favIconName = StringUtils.substringAfterLast(favIconURL.getPath(), "/");
            
            RequestConfig requestConfig = RequestConfig.custom()
                    .setConnectTimeout(_ICO_REQUEST_TIMEOUT)
                    .setConnectionRequestTimeout(_ICO_REQUEST_TIMEOUT)
                    .setSocketTimeout(_ICO_REQUEST_TIMEOUT)
                    .build();
            
            try (CloseableHttpClient httpclient = HttpClientBuilder.create()
                                                                   .setDefaultRequestConfig(requestConfig)
                                                                   .useSystemProperties()
                                                                   .setUserAgent(userAgent)
                                                                   .build())
            {
                // Prepare a request object
                HttpGet get = new HttpGet(favIconLink);
                try (CloseableHttpResponse httpResponse = httpclient.execute(get))
                {
                    int statusCode = httpResponse.getStatusLine().getStatusCode();
                    if (statusCode != 200)
                    {
                        getLogger().warn("Error " + statusCode + ". Can't set the favicon with url " + favIconLink + " to the link " + url);
                        return upload;
                    }
                    
                    try (InputStream is = httpResponse.getEntity().getContent())
                    {
                        HttpEntity entity = httpResponse.getEntity();
                        String mimeType = ContentType.get(entity).getMimeType();
                        if (mimeType.equals("image/x-icon") || mimeType.equals("image/vnd.microsoft.icon") || favIconName.endsWith(".ico"))
                        {
                            // If the image is .ico, convert it in .png
                            InputStream convertedIcoToPngIS = _urlPreviewComponent.convertIcoToPng(is);
                            upload = _uploadManager.storeUpload(user, StringUtils.replace(favIconName, ".ico", ".png"), convertedIcoToPngIS);
                        }
                        else if (_checkFavico(favIconName, mimeType))
                        {
                            upload = _uploadManager.storeUpload(user, favIconName, is);
                        }
                        else
                        {
                            getLogger().warn("Unsupported mime-type " + mimeType + " for favicon from url " + favIconLink + " to the link " + url);
                        }
                    }
                }
            }
            catch (IOException e)
            {
                getLogger().warn("Unable to download favicon from url " + favIconLink + " to the link " + url, e);
            }
        }
        catch (Exception e) 
        {
            getLogger().warn("Unable to get favicon from link url " + url, e);
        }
        
        return upload;
    }
    
    private boolean _checkFavico(String favicoName, String mimeType)
    {
        return __ICO_SUPPORTED_MIMETYPES.stream().anyMatch(m -> StringUtils.equalsIgnoreCase(m, mimeType))
                || __ICO_SUPPORTED_EXTENSIONS.stream().anyMatch(ext -> favicoName.toLowerCase().endsWith(ext));
        
    }
}
