/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * @private
 * Class defining a dialog box allowing to edit a shape of the {@link Ametys.plugins.maps.GMapPanel}
 */
Ext.define('Ametys.plugins.maps.EditShapeDialog', {
	extend: 'Ametys.window.DialogBox',

	/**
	 * @private
	 * @property {Function} _saveFn the function to invoke when validating the shapes' properties
	 */

	/**
	 * @private
	 * @property  {Ext.form.Panel} _form the form panel
	 */

	constructor: function(config)
	{
		this._saveFn = config.saveFn;

		this._form = this._createFormPanel();

		config = Ext.applyIf({
			title: "{{i18n PLUGINS_MAPS_SERVICE_CONFIGURATION_SHAPE_DIALOG_TITLE}}",
			
			width: 400,
			scrollable: false,
			
			items: [this._form],

			defaultFocus: 'title',
			closeAction: 'hide',
			
			referenceHolder: true,
			defaultButton: 'validate',
			
			buttons: [{
				reference: 'validate',
				text: "{{i18n PLUGINS_MAPS_SERVICE_CONFIGURATION_POI_SAVE}}",
				handler: this._ok,
				scope: this
			}, {
				text: "{{i18n PLUGINS_MAPS_SERVICE_CONFIGURATION_POI_CANCEL}}",
				handler: function () { this.hide(); },
				scope: this
			}]        
		}, config);

		this.callParent(arguments);
	},
	
	/**
	 * @private
	 * Initialize the form fields
	 * @param {String} title the title of the shape
	 * @param {String} description the description of the shape
	 * @param {String} color the code for the shape's color 
	 */
	_initForm: function(title, description, color)
	{
		var form = this._form.getForm();
		form.findField("title").setValue(title);
		form.findField("description").setValue(Ametys.convertHtmlToTextarea(description));
		form.findField("color").setValue(color);   
		
		form.clearInvalid();
		form.findField("title").focus();
	},
	
	/**
	 * @private
	 * Create the form panel
	 * @return the form panel for the edition of the shape properties
	 */
	_createFormPanel: function()
	{
		var form = Ext.create('Ext.form.Panel', {
			scrollable: true,
			border: false,
			
			fieldDefaults: {
		        labelAlign: 'right',
		        labelWidth: 100,
		        cls: 'ametys'
		    },

			defaultType: 'textfield',
			items: [
			        {
			        	fieldLabel: "{{i18n PLUGINS_MAPS_SERVICE_CONFIGURATION_POI_TITLE}}",                              
			        	name: 'title',
			        	itemId: 'title',
			        	width: 350,
			        	allowBlank: false
			        },
			        {
						xtype: 'textarea',
						fieldLabel: "{{i18n PLUGINS_MAPS_SERVICE_CONFIGURATION_POI_DESCRIPTION}}",
						width: 350,
			        	name: 'description'
			        },
			        {
			        	xtype: 'edition.colorpicker',
			        	fieldLabel: "{{i18n PLUGINS_MAPS_SERVICE_CONFIGURATION_POI_COLOR}}",
			        	name: 'color',
			        	width: 350,
                        allowBlank: false,
			        	'widget-params': {
			        		allowTransparent: true,
			        		allowOtherColors: true
						}
			        }
		      ]
		});

		return form;
	},
	
	/**
	 * @private
	 * Handler function invoked when clicking the 'ok' button
	 * Save the shape's properties if the form passes validation
	 */
	_ok: function()
	{
		var form = this._form.getForm();
		if (!form.isValid())
		{
			return;
		}

		var title = form.findField('title').getValue();
		var description = form.findField('description').getValue();
		var color = form.findField('color').getValue();

		if (Ext.isFunction(this._saveFn))
		{
			this._saveFn (title, description, color);
		}

		this.hide();
	}
});

