/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * @private
 * Class defining a dialog box allowing to edit a marker of the {@link Ametys.plugins.maps.GMapPanel}
 */
Ext.define('Ametys.plugins.maps.EditMarkerDialog', {
	extend: 'Ametys.window.DialogBox',
	
	/**
	 * @private
	 * @property {Function} _saveFn the function invoked to save the marker properties
	 */

	/**
	 * @private
	 * @property {Ametys.plugins.maps.MarkerIconsDialog} _iconsDialog the dialog allowing to select the marker's icon
	 */
	
	constructor: function(config)
	{
		this._form = this._createFormPanel();
		this._saveFn = config.saveFn;

		config = Ext.applyIf({
			title: "{{i18n PLUGINS_MAPS_SERVICE_CONFIGURATION_POI_DIALOG_TITLE}}",
			width: 370,
			
			closeAction: 'hide',
			
			scrollable: false,

			items: [this._form],
			
			defaultFocus: 'title',
			closeAction: 'hide',
			
			referenceHolder: true,
			defaultButton: 'validate',
			
			buttons: [{
				reference: 'validate',
				text: "{{i18n PLUGINS_MAPS_SERVICE_CONFIGURATION_POI_SAVE}}",
				handler: this._ok,
				scope: this
			}, {
				text: "{{i18n PLUGINS_MAPS_SERVICE_CONFIGURATION_POI_CANCEL}}",
				handler: function() {this.hide();},
				scope: this
			}]        
		}, config);

		this.callParent(arguments);
	},
	
	/**
	 * @private
	 * Create the form panel of the dialog
	 * @return the form panel of the dialog
	 */
	_createFormPanel: function()
	{
		var form = Ext.create('Ext.FormPanel', {
			scrollable: true,
			border: false,
			
			fieldDefaults: {
		        labelAlign: 'right',
		        labelWidth: 100,
		        cls: 'ametys'
		    },
			
			items: [
			        {
			        	fieldLabel: "{{i18n plugin.maps:PLUGINS_MAPS_SERVICE_CONFIGURATION_POI_TITLE}}",
			        	name: 'title',
			        	itemId: 'title',
			        	allowBlank: false,
			        	xtype: 'textfield',
			        	width: 300
			        },
			        {
			        	fieldLabel: "{{i18n plugin.maps:PLUGINS_MAPS_SERVICE_CONFIGURATION_POI_DESCRIPTION}}",
			        	name: 'description',
			        	xtype: 'textarea',
			        	width: 300
			        },
			        {
			        	fieldLabel: "{{i18n plugin.maps:PLUGINS_MAPS_SERVICE_CONFIGURATION_POI_ICON}}",
			        	itemId: 'markerField',
			        	xtype: 'fieldcontainer',
			        	items: [
		        	        {
					        	handler: Ext.bind(this._openMarkerIconsDialog, this),
		        	        	xtype: 'button',
		        	        	scale: 'large' // 32 * 32
		        	        }
	        			]
			        },
			        {                            
			        	name:'icon',
			        	xtype:'hidden'
			        }
			      ]
		});

		return form;
	},
	
	/**
	 * @private
	 * Initialize the dialok box
	 * @param {String} title the title chosen for the marker
	 * @param {String} description the description chosen for the marker
	 * @param {String} icon the name of the icon chosen to represent the marker
	 */
	_initForm: function(title, description, icon)
	{
		var form = this._form.getForm();
		form.findField("title").setValue(title);
		form.findField("description").setValue(Ametys.convertHtmlToTextarea(description));
		form.findField("icon").setValue(icon || 'pin');   

		var iconBtn = this._form.getComponent('markerField');
		iconBtn.items.get(0).setIcon(Ametys.getPluginResourcesPrefix('maps') + '/img/poi/' + (icon || 'pin') + '.png'); 

		form.findField("title").focus();
		form.clearInvalid();
	},
	
	/**
	 * @private
	 * Open the marker icons dialog
	 */
	_openMarkerIconsDialog: function()
	{
		if (!this._iconsDialog)
		{
			this._iconsDialog = Ext.create ('Ametys.plugins.maps.MarkerIconsDialog', {saveFn: Ext.bind(this._chooseIcon, this)});
		}

		var icon = this._form.getForm().findField("icon").getValue();

		this._iconsDialog._loadIcons(icon); 
		this._iconsDialog.show();
	},
	
	/**
	 * @private
	 * Function invoked when a marker has been selected and validated in the {@link Ametys.plugins.maps.MarkerIconsDialog} 
	 * @param {String} icon the name of the icon 
	 */
	_chooseIcon: function (icon)
	{
		if (icon)
		{
			this._form.getForm().findField("icon").setValue(icon);
			
			var iconBtn = this._form.getComponent('markerField');
			iconBtn.items.get(0).setIcon(Ametys.getPluginResourcesPrefix('maps') + '/img/poi/' + icon + '.png'); 
		}
	},
	
	/**
	 * @private
	 * Function invoked when clicking 'ok', allowing to the save the marker's properties
	 */
	_ok: function()
	{
		var form = this._form.getForm();
		if (!form.isValid())
		{
			return;
		}

		var title = form.findField('title').getValue();
		var description = form.findField('description').getValue();
		var icon = form.findField('icon').getValue();

		if (Ext.isFunction(this._saveFn))
		{
			this._saveFn (title, description, icon);
		}

		this.hide();
	}
	
});

