/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * @private
 * Google Map configuration object. It stores the data shared by the {@link Ametys.plugins.maps.MapItemsView} and the {@link Ametys.plugins.maps.GMapPanel}. 
 */
Ext.define('Ametys.plugins.maps.GMapConfiguration', {

	/**
	 * @private
	 * @property {Ametys.plugins.maps.GMapPanel} _gMapPanel the panel of the google map
	 */

	/**
	 * @private
	 * @property {Ametys.plugins.maps.MapItemsView} _mapItemsView the data view for the items of the map
	 */

	constructor: function(config)
	{
		Ext.apply(this, config);
		
		Ext.applyIf(this, {
			mapTypeId: google.maps.MapTypeId.ROADMAP,
			zoomLevel: 1,
			center: {
				lat:0.0,
				lng:0.0
			},
			pois : []
		});

		Ext.applyIf(this, {
			store: Ext.create ('Ext.data.JsonStore', {
				model: Ametys.plugins.maps.GMapConfiguration.GMapElement,
				sorters: [{ property: 'gtype', direction: "ASC" }],
				data: this.pois
			})
		});

		this.callParent(arguments);
	},

	/**
	 * Get the data view for registered POIs
	 * @return {Ext.panel.Panel} The map items panel
	 */
	getMapItemsPanel: function (config)
	{
		var viewConfig = 
		{
				prepareData: function (data, recordIdx, record) 
				{ 
					data.shortTitle = Ext.util.Format.ellipsis(data.title, 30, false);  
					data.recordId = record.id;
					return data;
				},
				store: this.store
		};

		this._mapItemsView = Ext.create('Ametys.plugins.maps.MapItemsView', viewConfig);

		var panelConfig = Ext.applyIf (config,  
		{
				title: "{{i18n PLUGINS_MAPS_SERVICE_CONFIGURATION_POIS}}",
				scrollable: true,
                flex: 1,
				border: false,
				cls: 'poi-fieldset',
				bodyStyle: {
					padding: '5px'
				},
				collapsible: false,
				titleCollapse: false,
				items: [this._mapItemsView]
		});

		return Ext.create('Ext.panel.Panel', panelConfig);
	},

	/**
	 * Get the {@link Ametys.plugins.maps.GMapPanel} for the service
	 * @return {Ametys.plugins.maps.GMapPanel} the panel
	 */
	getGMapPanel: function (config)
	{
		var panelConfig = Ext.applyIf(config,
			{ 
				zoomLevel: this.zoomLevel,
				center: this.center,
				pois: this.pois,

				mapTypeId: this.mapTypeId,
				store: this.store        			
			}
		); 
		
		this._gMapPanel = Ext.create('Ametys.plugins.maps.GMapPanel', panelConfig);
		return this._gMapPanel;
	},

	/**
	 * Get the center of the map
	 * @return {Object} the center of the map
	 */
	getCenter: function()
	{
		var center = {
			lat: 0.0,
			lng: 0.0
		};
		
		if (this._gMapPanel)
		{
			var lastCenter = this._gMapPanel.getLastCenter();
			if (lastCenter)
			{
				center.lat = lastCenter.lat();
				center.lng = lastCenter.lng();
			}            
		}

		return center;
	},

	/**
	 * 
	 * Handles click on the edit image of the map items panel
	 * @param {String} recordId the id of the record to edit 
	 */
	editItemAtIndex: function(recordId)
	{    
		var record = this.store.getById(recordId);
		if(record)
		{            
			if (record.get('gtype') == 'marker')
			{
				this._gMapPanel._editMarkerProperties(record);
			}
			else if(record.get('gtype') == 'polygon')
			{
				this._gMapPanel._editShapeProperties(record);
			}
		}
		return;
	},

	/**
	 * Handles click on the delete image of the map items panel
	 * @param {String} recordId the id of the record to delete
	 */
	removeItemAtIndex: function(recordId)
	{
		var record = this.store.getById(recordId);      	  
		if (record)
		{
			this.store.remove(record);
		}
		return;
	},

	/**
	 * Get the zoom level of the map
	 * @return the zoom level of the map
	 */
	getZoomLevel: function()
	{
		var zoomLevel = 1;
		if (this._gMapPanel)
		{
			zoomLevel = this._gMapPanel.getZoomLevel();           
		}
		return zoomLevel;
	},

	/**
	 * Get the id of the map type
	 * @return the id of the map type
	 */
	getMapTypeId: function()
	{
		var mapType = google.maps.MapTypeId.ROADMAP;
		if (this._gMapPanel)
		{
			mapType = this._gMapPanel.getMapTypeId();           
		}
		return mapType;
	},

	/**
	 * Get the points of interest as a JSON string
	 * @return the points of interest as a JSON string
	 */
	getPoisAsArray: function()
	{
		var poisAsArray = [];
		if (this.store)
		{
			this.store.each( function(record)
			{
				// only serialize needed properties (ex recordId is not serialized)
				poisAsArray.push(Ext.copyTo({},record.data,"lat,lng,title,description,color,icon,points,gtype"));
			});
		}
		
		return poisAsArray;
	}
});

