/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.messagingconnector;

import org.ametys.plugins.linkdirectory.dynamic.DynamicInformationException;

/**
 * Common exception for all messaging connectors.
 */
public class MessagingConnectorException extends RuntimeException
{
    /**
     * Information about the exception
     */
    public enum ExceptionType
    {
        /** The connexion is not authorized for this specific user*/
        UNAUTHORIZED,
        /** Connexion timed out */
        TIMEOUT,
        /** Configuration exception (unknown host, host unreachable, 404, ...) */
        CONFIGURATION_EXCEPTION,
        /** Nothing defined */
        UNKNOWN;
    }
    
    private ExceptionType _type = ExceptionType.UNKNOWN;
    /**
     * Constructs a new exception with the specified detail message.
     * @param message the detail message.
     */
    public MessagingConnectorException(String message)
    {
        this(message, ExceptionType.UNKNOWN);
    }
    /**
     * Constructs a new exception with the specified detail message.
     * @param message the detail message.
     * @param type reason of the exception
     */
    public MessagingConnectorException(String message, ExceptionType type)
    {
        super(message);
        this._type = type;
    }

    /**
     * Constructs a new exception with the specified detail message and cause.
     * @param message the detail message.
     * @param cause the cause.
     */
    public MessagingConnectorException(String message, Throwable cause)
    {
        this(message, ExceptionType.UNKNOWN, cause);
    }
    
    /**
     * Constructs a new exception with the specified detail message and cause.
     * @param message the detail message.
     * @param type reason of the exception
     * @param cause the cause.
     */
    public MessagingConnectorException(String message, ExceptionType type, Throwable cause)
    {
        super(message, cause);
        this._type = type;
    }

    /**
     * Constructs a new exception with the specified cause.
     * @param cause the specified cause.
     */
    public MessagingConnectorException(Throwable cause)
    {
        this(ExceptionType.UNKNOWN, cause);
    }

    /**
     * Constructs a new exception with the specified cause.
     * @param type reason of the exception
     * @param cause the specified cause.
     */
    public MessagingConnectorException(ExceptionType type, Throwable cause)
    {
        super(cause);
        this._type = type;
    }
    
    /**
     * Get the type of exception
     * @return {@link ExceptionType}
     */
    public ExceptionType getType()
    {
        return _type;
    }
    /**
     * Create a DynamicInformationException from this MessagingConnectorException
     * @return a new DynamicInformationException
     */
    public DynamicInformationException toDynamicInformationException()
    {
        DynamicInformationException.ExceptionType type = DynamicInformationException.ExceptionType.UNKNOWN;
        switch (getType())
        {
            case CONFIGURATION_EXCEPTION:
                type = DynamicInformationException.ExceptionType.CONFIGURATION_EXCEPTION;
                break;
            case TIMEOUT:
                type = DynamicInformationException.ExceptionType.TIMEOUT;
                break;
            case UNAUTHORIZED:
                type = DynamicInformationException.ExceptionType.UNAUTHORIZED;
                break;
            case UNKNOWN:
            default:
                type = DynamicInformationException.ExceptionType.UNKNOWN;
                break;
        }
        
        return new DynamicInformationException(getMessage(), type, this);
    }
}
