/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.messagingconnector.dynamic;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.plugins.linkdirectory.dynamic.AbstractInternalDynamicInformationGenerator;
import org.ametys.plugins.linkdirectory.dynamic.DynamicInformationException;
import org.ametys.plugins.linkdirectory.dynamic.DynamicInformationException.ExceptionType;
import org.ametys.plugins.messagingconnector.MessagingConnector;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.transformation.xslt.AmetysXSLTHelper;

/**
 * Abstract generator for saxing dynamic information for agenda/mail service
 */
public abstract class AbstractMessagingConnectorDynamicInformationGenerator extends AbstractInternalDynamicInformationGenerator
{
    /** The messagingConnector component */
    protected MessagingConnector _messagingConnector;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _messagingConnector = (MessagingConnector) manager.lookup(MessagingConnector.ROLE);
    }
    
    @Override
    protected void saxError(DynamicInformationException e) throws SAXException
    {
        if (e.getType() == ExceptionType.UNAUTHORIZED && _messagingConnector.supportUserCredential(getCurrentUser()))
        {
            AttributesImpl attr = new AttributesImpl();
            attr.addCDATAAttribute(ERROR_ATTRIBUTE_TYPE, e.getType().name());
            XMLUtils.startElement(contentHandler, ERROR, attr);
            
            // Error message
            I18nizableText errorMessage = new I18nizableText("plugin.link-directory", "PLUGINS_LINKDIRECTORY_DYNAMIC_INFO_PROVIDER_ERROR_" + ExceptionType.UNAUTHORIZED.name());
            errorMessage.toSAX(contentHandler);
            
            String spanId = getSpanId();
            
            // span element where to insert change password link
            attr.clear();
            attr.addCDATAAttribute("id", spanId);
            attr.addCDATAAttribute("class", "link-change-password-link");
            XMLUtils.createElement(contentHandler, "span", attr);
            
            // JS script to change password
            attr.clear();
            attr.addCDATAAttribute("type", "text/javascript");
            XMLUtils.startElement(contentHandler, "script", attr);
            XMLUtils.data(contentHandler, _scriptToUpdatePassword(spanId));
            XMLUtils.endElement(contentHandler, "script");
            
            XMLUtils.endElement(contentHandler, ERROR);
        }
        else
        {
            super.saxError(e);
        }
    }
    
    /**
     * Get the id of span element where to insert the link to set/update password
     * @return id of span element
     */
    protected abstract String getSpanId();

    private String _scriptToUpdatePassword(String spanId)
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String uriPrefix = AmetysXSLTHelper.uriPrefix();
        String siteUriPrefix = AmetysXSLTHelper.siteUriPrefix();
        String lang = request.getParameter("lang");
        String uniqueId = request.getParameter("uniqueId");
        
        return "if (!window.MessagingConnector || !MessagingConnector.insertChangePasswordLink)\n"
                    + "{\n"
                    + String.format("\tMessagingConnector = {SITE_URI_PREFIX: '%s', LANG: '%s'}\n", siteUriPrefix, lang)
                    + String.format("\t$j.getScript(\"%s/plugins/messaging-connector/resources/js/messaging-connector-password.js\")", uriPrefix, lang)
                    + String.format(".done(function(){MessagingConnector.insertChangePasswordLink('%s', updateDynamicInformation_%s)})", spanId, uniqueId)
                    + ".fail(function(jqxhr, settings, exception) { console.error(\"An error occurred while loading js files\", exception); })\n"
                    + "}\nelse{\n"
                    + String.format("\tMessagingConnector.insertChangePasswordLink('%s', updateDynamicInformation_%s)\n", spanId, uniqueId)
                    + "}\n";
                                        
    }

}
