/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.messagingconnector.dynamic;

import java.util.Collections;
import java.util.List;

import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.StringUtils;
import org.ametys.plugins.linkdirectory.dynamic.DynamicInformationException;
import org.ametys.plugins.messagingconnector.CalendarEvent;
import org.ametys.plugins.messagingconnector.MessagingConnectorException;
import org.ametys.runtime.i18n.I18nizableText;

/**
 *  Generates sax events for calendar information 
 */
public class CalendarDynamicInformationGenerator extends AbstractMessagingConnectorDynamicInformationGenerator
{
    private static final int __DEFAULT_MAX_DAYS = 7;
    
    @Override
    protected void saxShortValue() throws SAXException, DynamicInformationException
    {
        try
        {
            UserIdentity currentUser = getCurrentUser();
            if (currentUser != null)
            {
                int eventsCount = _messagingConnector.getEventsCount(currentUser, getMaxDays());
                XMLUtils.createElement(contentHandler, SHORT_VALUE, String.valueOf(eventsCount));
            }
            else
            {
                XMLUtils.createElement(contentHandler, SHORT_VALUE, "0");
            }
        }
        catch (MessagingConnectorException e)
        {
            throw e.toDynamicInformationException();
        }
    }

    @Override
    protected void saxLongValue() throws SAXException, DynamicInformationException
    {
        try
        {
            UserIdentity currentUser = getCurrentUser();
            if (currentUser != null)
            {
                int eventsCount = _messagingConnector.getEventsCount(currentUser, getMaxDays());
                
                String i18nKey = eventsCount > 1 ? "PLUGINS_MESSAGINGCONNECTOR_LINKDIRECTORY_CALENDAR_DISPLAY_MULTIPLE" : "PLUGINS_MESSAGINGCONNECTOR_LINKDIRECTORY_CALENDAR_DISPLAY_SINGLE";
                
                I18nizableText longValue = new I18nizableText("plugin.messaging-connector", i18nKey, Collections.singletonList(String.valueOf(eventsCount)));
                longValue.toSAX(contentHandler, LONG_VALUE);
            }
            else
            {
                I18nizableText longValue = new I18nizableText("plugin.messaging-connector", "PLUGINS_MESSAGINGCONNECTOR_LINKDIRECTORY_CALENDAR_NO_USER");
                longValue.toSAX(contentHandler, LONG_VALUE);
            }
        }
        catch (MessagingConnectorException e)
        {
            throw e.toDynamicInformationException();
        }
    }

    @Override
    protected void saxTooltips() throws SAXException, DynamicInformationException
    {
        try
        {
            UserIdentity currentUser = getCurrentUser();
            if (currentUser != null)
            {
                List<CalendarEvent> events = _messagingConnector.getEvents(currentUser, getMaxDays(), getMaxItems());
                
                for (CalendarEvent calendarEvent : events)
                {
                    saxItem(calendarEvent.getStartDate(), calendarEvent.getSubject(), calendarEvent.getLocation());
                }
            }
        }
        catch (MessagingConnectorException e)
        {
            throw e.toDynamicInformationException();
        }
    }
    
    /**
     * Get the max days for search
     * @return the max number of day
     */
    protected int getMaxDays() 
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        try
        {
            return Integer.parseInt(request.getParameter("maxDays"));
        }
        catch (NumberFormatException e)
        {
            return __DEFAULT_MAX_DAYS;
        }
    }
    
    @Override
    protected String getSpanId()
    {
        return "dynamic-info-calendar-change-password-" + StringUtils.generateKey();
    }
}
