/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.messagingconnector;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.messagingconnector.MessagingConnectorException.ExceptionType;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * 
 * This class is the action used by the messaging connector plugin.
 * It uses the correct mail server component.
 *
 */
public class MessagingConnectorAction extends ServiceableAction
{
    /** Date format cache for different locale (date and time) */
    protected static final Map<Locale, DateFormat> __DATETIME_FORMATS = new HashMap<>();
    
    /** Date format cache for different locale (time only) */
    protected static final Map<Locale, DateFormat> __TIME_FORMATS = new HashMap<>();
    
    /** The output date format. */
    protected static final DateFormat DATE_FORMAT = new SimpleDateFormat("EEE d MMM H'h'mm");
    
    /** The max days parameter from the URL */
    private static final String MAX_DAYS = "maxDays";

    /** The max events parameter from the URL */
    private static final String MAX_EVENTS = "maxEvents";
    
    /** mailServer selected */
    protected String _mailServerId;

    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** The user manager */
    protected UserManager _usersManager;

    /** The messaging connector that will be used*/
    protected MessagingConnector _messagingConnector;

    /** The I18N utils */
    protected I18nUtils _i18nUtils;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _messagingConnector = (MessagingConnector) smanager.lookup(MessagingConnector.ROLE);
        _usersManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }

    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters)
    {
        Map<String, Object> result = new HashMap<>();
        Request request = ObjectModelHelper.getRequest(objectModel);
        Boolean success = true;
        try
        {
            UserIdentity identity = _currentUserProvider.getUser();
            if (identity != null)
            {
                result.put("unreadMessages", _messagingConnector.getUnreadEmailCount(identity));
                
                int maxDays = Integer.parseInt(request.getParameter(MAX_DAYS));
                int maxEvents = Integer.parseInt(request.getParameter(MAX_EVENTS));
                
                List<CalendarEvent> nextEvents = _messagingConnector.getEvents(identity, maxDays, maxEvents);
                
                result.put("events", eventsToJson(nextEvents, getLocale(request, objectModel)));
                result.put("nbNextEvents", _messagingConnector.getEventsCount(identity, maxDays));
            }
        }
        catch (MessagingConnectorException e)
        {
            success = false;
            ExceptionType type = e.getType();
            switch (type)
            {
                case TIMEOUT:
                case UNAUTHORIZED:
                case CONFIGURATION_EXCEPTION:
                    result.put("error", type.name());
                    I18nizableText errorMessage = new I18nizableText("plugin.messaging-connector", "PLUGINS_MESSAGINGCONNECTOR_ERROR_" + type.name().toUpperCase());
                    result.put("message", errorMessage);
                    break;
                case UNKNOWN:
                default:
                    throw e;
            }
        }

        result.put("success", success);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    /**
     * Get the current locale
     * @param request The request
     * @param objectModel The object model
     * @return the locale to use
     */
    protected Locale getLocale(Request request, Map objectModel)
    {
        String lang = request.getParameter("lang");
        if (StringUtils.isNotEmpty(lang))
        {
            return LocaleUtils.toLocale(lang);
        }
        else
        {
            return org.apache.cocoon.i18n.I18nUtils.findLocale(objectModel, "locale", null, Locale.getDefault(), true);
        }
    }
    
    /**
     * This methods is used to create a List with all the correct informations in order to be displayed by the plugin.
     * 
     * @param events the events to be displayed
     * @param locale the locale of the user
     * @return A list of Maps destined for Json
     */
    protected List<Map<String, Object>> eventsToJson(List<CalendarEvent> events, Locale locale) 
    {
        List<Map<String, Object>> eventsDisplayed = new ArrayList<>();
        
        for (CalendarEvent event : events)
        {
            Map<String, Object> eventDisplayed = new HashMap<>();
            eventDisplayed.put("eventStartDateFormatted", _getFormattedDate(event.getStartDate(), locale));
            eventDisplayed.put("eventEndDateFormatted", _getFormattedDate(event.getEndDate(), locale));
            eventDisplayed.put("eventLocation", event.getLocation());
            eventDisplayed.put("eventSubject", event.getSubject());

            eventsDisplayed.add(eventDisplayed);
        }

        return eventsDisplayed;
    }
    
    /**
     * Format a Date to a String using a locale to set the correct date
     * 
     * @param date the date to format
     * @param locale the locale of the user
     * @return The date formated with the given locale
     */
    protected String _getFormattedDate(Date date, Locale locale)
    {
        // -- START DATE --
        // Retrieves the desired date format for the current locale
        boolean isSameDay = org.apache.commons.lang3.time.DateUtils.isSameDay(date, new Date());
        Map<Locale, DateFormat> dateFormats = isSameDay ? __TIME_FORMATS : __DATETIME_FORMATS;
        
        DateFormat df = dateFormats.get(locale);
        if (df == null)
        {
            String key = isSameDay ? "PLUGINS_MESSAGINGCONNECTOR_RDV_TIME_FORMAT" : "PLUGINS_MESSAGINGCONNECTOR_RDV_DATETIME_FORMAT";
            String strFormat = _i18nUtils.translate(new I18nizableText("plugin.messaging-connector", key), locale.toString());
            df = new SimpleDateFormat(strFormat, locale);
            dateFormats.put(locale, df);
        }
        
        return df.format(date);
    }
}
