/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.messagingconnector;

import java.time.ZonedDateTime;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.component.Component;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;

/**
 * This interface describe the methods used by the different mail servers.
 *
 */
public interface MessagingConnector extends Component
{
    /** The avalon role */
    public static final String ROLE = MessagingConnector.class.getName();

    /**
     * Get the list of population id allowed to the messaging connector
     * @return the list of population id allowed to the messaging connector
     * @throws MessagingConnectorException if failed to get populations
     */
    public List<String> getAllowedPopulationIds() throws MessagingConnectorException;
    
    /**
     * Get the list of upcoming events from now for a given user
     * @param userIdentity the owner of the events
     * @param maxDays The maximum number of days to search for from now 
     * @param maxEvents the maximum number of events to retrieve
     * @return the next events
     * @throws MessagingConnectorException if failed to get events
     */
    public List<CalendarEvent> getEvents(UserIdentity userIdentity, int maxDays, int maxEvents) throws MessagingConnectorException;

    /**
     * Return the number of upcoming events
     * @param userIdentity the owner of the events
     * @param maxDays The maximum number of days to search for from now 
     * @return the number of upcoming events
     * @throws MessagingConnectorException if failed to get events' count
     */
    public int getEventsCount(UserIdentity userIdentity, int maxDays) throws MessagingConnectorException;

    /**
     * Return the unread emails for a given user
     * 
     * @param userIdentity the recipient of the mails
     * @param maxEmails The max number of emails to return
     * @return a mail messsages
     * @throws MessagingConnectorException if failed to get mails
     */
    public List<EmailMessage> getUnreadEmails(UserIdentity userIdentity, int maxEmails) throws MessagingConnectorException;

    /**
     * Return the number of unread mails for a given user
     * @param userIdentity the receiver of the mails
     * @return the number of unread mails
     * @throws MessagingConnectorException if failed to get unread mails ' count
     */
    public int getUnreadEmailCount(UserIdentity userIdentity) throws MessagingConnectorException;
    
    /**
     * Tell if the messaging connector supports event invitation
     * @return true if the messaging connector supports event invitation
     * @throws MessagingConnectorException if failed to get events from server
     */
    public abstract boolean supportInvitation() throws MessagingConnectorException;
    
    /**
     * True if the event exist in the messaging connector
     * @param eventId the event id
     * @param organiser the organiser
     * @return true if the event exist
     * @throws MessagingConnectorException if an error occurred
     */
    public abstract boolean isEventExist(String eventId, UserIdentity organiser) throws MessagingConnectorException;
    
    /**
     * Create an event
     * @param title the event title
     * @param description the event description
     * @param place the event place
     * @param isAllDay if the event is all day
     * @param startDate the event start date
     * @param endDate the event end date
     * @param recurrenceType recurrence type
     * @param untilDate until date of the recurring event
     * @param attendees the map of attendees (email -&gt; optional or requested) to set
     * @param organiser the event organiser
     * @return the id of the event created
     * @throws MessagingConnectorException if failed to get events from server
     */
    public abstract String createEvent(String title, String description, String place, boolean isAllDay, ZonedDateTime startDate, ZonedDateTime endDate, EventRecurrenceTypeEnum recurrenceType, ZonedDateTime untilDate, Map<String, Boolean> attendees, UserIdentity organiser) throws MessagingConnectorException;
    
    /**
     * Update an event
     * @param eventId the event id to delete
     * @param title the event title
     * @param description the event description
     * @param place the event place
     * @param isAllDay if the event is all day
     * @param startDate the event start date
     * @param endDate the event end date
     * @param recurrenceType recurrence type
     * @param untilDate until date of the recurring event
     * @param attendees the map of attendees (email -&gt; optional or requested) to set
     * @param organiser the event organiser
     * @throws MessagingConnectorException if failed to get events from server
     */
    public abstract void updateEvent(String eventId, String title, String description, String place, boolean isAllDay, ZonedDateTime startDate, ZonedDateTime endDate, EventRecurrenceTypeEnum recurrenceType, ZonedDateTime untilDate, Map<String, Boolean> attendees, UserIdentity organiser) throws MessagingConnectorException;
    
    /**
     * Delete an event
     * @param eventId the event id to delete
     * @param organiser the event organiser
     * @throws MessagingConnectorException if failed to get events from server
     */
    public abstract void deleteEvent(String eventId, UserIdentity organiser) throws MessagingConnectorException;
    
    /**
     * Get the map of attendees for an event
     * @param eventId the event id
     * @param organiser the event organiser
     * @return the map of attendees (email -&gt; attendee information)
     * @throws MessagingConnectorException if failed to get events from server
     */
    public abstract Map<String, AttendeeInformation> getAttendees(String eventId, UserIdentity organiser) throws MessagingConnectorException;
    
    /**
     * Set attendees for an event
     * @param eventId the event id
     * @param attendees the map of attendees (email -&gt; optional or requested) to set
     * @param organiser the event organiser
     * @throws MessagingConnectorException if failed to get events from server
     */
    public abstract void setAttendees(String eventId, Map<String, Boolean> attendees, UserIdentity organiser) throws MessagingConnectorException;
    
    /**
     * Get free/busy status for attendees for a time window
     * @param startDate the start date
     * @param endDate the end date
     * @param isAllDay true if is an allday event
     * @param attendees the list of attendees email
     * @param organiser the event organiser
     * @return the map of attendees (email -&gt; freeBusy status)
     * @throws MessagingConnectorException if failed to get events from server
     */
    public abstract Map<String, FreeBusyStatus> getFreeBusy(Date startDate, Date endDate, boolean isAllDay, Set<String> attendees, UserIdentity organiser) throws MessagingConnectorException;
    
    /**
     * True if the user exist
     * @param userIdentity the user identity
     * @return true if the user exist
     * @throws MessagingConnectorException if failed to get events from server
     */
    public abstract boolean isUserExist(UserIdentity userIdentity) throws MessagingConnectorException;
    
    /**
     * Check if the service needs user credential (some implementation can use a central API key or impersonation, some don't)
     * @param userIdentity The useridentity involved in the demand
     * @return True if the service is configured to ask for user credential
     */
    public abstract boolean userCredentialNeeded(UserIdentity userIdentity);
    
    /**
     * Check if a user credential should be saved
     * @param userIdentity The useridentity involved in the demand
     * @return true if this implementation supports user credential (and is configured to use them)
     */
    public abstract boolean supportUserCredential(UserIdentity userIdentity);

    /**
     * Set a new password for the user
     * @param userIdentity user that have a password change
     * @param password new password
     * @throws UserPreferencesException An exception occured while saving the password
     * @throws MessagingConnectorException Thrown when the messaging connector does not support user credentials
     */
    public void setUserPassword(UserIdentity userIdentity, String password) throws UserPreferencesException, MessagingConnectorException;
    
    /**
     * Enum for free/busy status
     *
     */
    public enum FreeBusyStatus
    {
        /**
         * The attendee is busy
         */
        Busy,
        /**
         * The attendee is free
         */
        Free,
        /**
         * We don't know if the attendee is busy or free
         */
        Unknown;
    }
    
    /**
     * Enum for answer status
     *
     */
    public enum ResponseType
    {
        /**
         * The attendee accepts the invitation
         */
        Accept,
        /**
         * The attendee declines the invitation
         */
        Decline,
        /**
         * The attendee says maybe to the event
         */
        Maybe,
        /**
         * We don't know the answer of the attendee
         */
        Unknown;
    }
    
    /**
     * Internal class for attendee information
     *
     */
    public class AttendeeInformation
    {
        private boolean _mandatory;
        private ResponseType _responseType;
        
        /**
         * Constructor for an attendee information
         * @param mandatory mandatory
         * @param responseType the answer type
         */
        public AttendeeInformation (boolean mandatory, ResponseType responseType)
        {
            _mandatory = mandatory;
            _responseType = responseType;
        }
        
        /**
         * True if the attendee is mandatory
         * @return true if the attendee is mandatory
         */
        public boolean isMandatory ()
        {
            return _mandatory;
        }
        
        /**
         * Get the response type
         * @return the response type
         */
        public ResponseType getResponseType ()
        {
            return _responseType;
        }
    }
}
