/*
 *  Copyright 2026 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.site;

import java.util.List;

import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.Session;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;

import org.ametys.plugins.site.proxy.BackOfficeRequestProxy;
import org.ametys.plugins.site.proxy.SessionAttributeRequestProxy;

/**
 * Request proxy to forward OIDC tokens from the mobile app to the back office.
 */
public class OIDCSessionAttributeRequestProxy implements BackOfficeRequestProxy
{
    /** The prefix for incoming request headers from mobile app */
    public static final String HEADER_PREFIX = "X-Ametys-Mobile-";
    
    private static final List<String> __ATTRIBUTES = List.of("oidc_token", "msal_token");

    @Override
    public void prepareBackOfficeRequest(Request request, HttpUriRequest backOfficeRequest)
    {
        Session session = request.getSession(false);
        if (session != null)
        {
            for (String attributeName : __ATTRIBUTES)
            {
                // First, check if the attribute is present in the session
                Object attribute = session.getAttribute(attributeName);
                
                // Otherwise, check if it's present as a request header (pushed by the mobile app)
                if (attribute == null)
                {
                    attribute = request.getHeader(HEADER_PREFIX + attributeName);
                    
                    // FIXME MOBILEAPP-103: store session attributes at authentication time
                    if (attribute != null)
                    {
                        session.setAttribute(attributeName, attribute);
                    }
                }
                
                if (attribute != null)
                {
                    backOfficeRequest.setHeader(SessionAttributeRequestProxy.HEADER_PREFIX + attributeName, attribute.toString());
                }
            }
        }
    }
    
    public void handleBackOfficeResponse(Response response, HttpResponse backOfficeResponse)
    {
        // Nothing to do
    }
}
