/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.project.ProjectsCatalogueManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.plugin.PluginsManager;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper to handle project feeds
 */
public class FeedHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = FeedHelper.class.getName();
    
    /** I18N Utils */
    protected I18nUtils _i18nUtils;
    
    /** The project catalogue manager component */
    protected ProjectsCatalogueManager _projectsCatalogueManager;

    public void service(ServiceManager manager) throws ServiceException
    {
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        if (PluginsManager.getInstance().isPluginActive("workspaces"))
        {
            _projectsCatalogueManager = (ProjectsCatalogueManager) manager.lookup(ProjectsCatalogueManager.ROLE);
        }
    }
    

    /**
     * Generate a map of project names with the basic projects informations to return
     * @return a map of basic information about projects
     */
    public Map<String, Map<String, Object>> getProjects()
    {
        Map<String, Map<String, Object>> result = new HashMap<>();
        
        if (_projectsCatalogueManager != null)
        {
            List<Map<String, Object>> userProjects = _projectsCatalogueManager.getUserProjects();
            for (Map<String, Object> fullProjectMap : userProjects)
            {
                String name = (String) fullProjectMap.get("name");
                Map<String, Object> neededInfos = new HashMap<>();
    
                neededInfos.put("name", name);
                neededInfos.put("description", fullProjectMap.get("description"));
                neededInfos.put("title", fullProjectMap.get("title"));
                neededInfos.put("url", fullProjectMap.get("url"));
                neededInfos.put("illustration", fullProjectMap.get("illustration"));
                neededInfos.put("language", fullProjectMap.get("language"));
                neededInfos.put("id", fullProjectMap.get("id"));
                neededInfos.put("category", fullProjectMap.get("category"));
               
                result.put(name, neededInfos);
            }
        }
        
        return result;
    }
    
    /**
     * Transform a {@link Project} into a json map
     * @param project the project to parse
     * @return a json map
     */
    public Map<String, Object> projectToMap(Project project)
    {
        return _projectsCatalogueManager.detailedProject2json(project);
    }
    /**
     * Add infos to the activity, so it can be displayed by the app
     * @param activity the json representing the activity to parse
     * @param project the project, as a json map (see {@link FeedHelper#projectToMap(Project)})
     * @param lang language to use to translate the short description
     * @return a map to return in json
     */
    public Map<String, Object> getActivityInfos(Map<String, Object> activity, Map<String, Object> project, String lang)
    {
        Map<String, Object> result = new HashMap<>();
        
        result.putAll(activity);
        
        result.put("project", project);
        
        @SuppressWarnings("unchecked")
        Map<String, String> activityAuthor = (Map<String, String>) activity.get("author");
        
        Map<String, String> author = new HashMap<>();
        author.put("fullname", activityAuthor.get("fullname"));
        result.put("author", author);
        
        result.put("short-description", getActivityDescription(activity, lang));

        result.put("content_id", getActivityObjectId(activity));
        result.put("content_url", getActivityUrl(activity));
        
        return result;
    }
    
    /**
     * Generate a description for this activity
     * @param activity the activity to describe
     * @return a String of the description
     */
    @SuppressWarnings("unchecked")
    protected String getActivityObjectId(Map<String, Object> activity)
    {
        String id = null;
        String eventType = (String) activity.get("type");
        
        switch (eventType)
        {
            /*
             * RESOURCES
             */
            case org.ametys.plugins.explorer.ObservationConstants.EVENT_RESOURCE_CREATED :
            case org.ametys.plugins.explorer.ObservationConstants.EVENT_RESOURCE_UPDATED :
            case org.ametys.plugins.explorer.ObservationConstants.EVENT_RESOURCE_RENAMED :
                if (activity.containsKey("file"))
                {
                    id = (String) ((Map<String, Object>) activity.get("file")).get("id");
                }
                break;

            /*
             * CALENDAR
             */
            case org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_EVENT_CREATED :
            case org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_EVENT_UPDATED :
                id = (String) activity.get("eventId");
                break;

            /*
             * THREAD
             */
            case ObservationConstants.EVENT_THREAD_CREATED :
            case ObservationConstants.EVENT_THREAD_COMMENTED :
                id = (String) activity.get("threadId");
                break;

            /*
             * MEMBER
             */
            case ObservationConstants.EVENT_MEMBER_ADDED :
                // no id available
                break;
            case ObservationConstants.EVENT_WALLCONTENT_ADDED :
                id = (String) activity.get("contentId");
                break;

            /**
             * WIKI
             */
            case ObservationConstants.EVENT_MINISITE_PAGE_CREATED :
            case ObservationConstants.EVENT_MINISITE_PAGE_UPDATED :
            case ObservationConstants.EVENT_MINISITE_PAGE_RENAMED :
            case ObservationConstants.EVENT_MINISITE_PAGE_DELETED :
                id = (String) activity.get("pageId");
                break;

            /**
             * TASK
             */
            case ObservationConstants.EVENT_TASK_CREATED :
            case ObservationConstants.EVENT_TASK_ASSIGNED :
            case ObservationConstants.EVENT_TASK_CLOSED_STATUS_CHANGED :
                id = (String) activity.get("taskId");
                break;
            default:
                break;
        }
        
        return id;
    }
    

    /**
     * Generate a description for this activity
     * @param activity the activity to describe
     * @return a String of the description
     */
    protected String getActivityUrl(Map<String, Object> activity)
    {
        String url = null;
        String eventType = (String) activity.get("type");
        
        switch (eventType)
        {
            /*
             * RESOURCES
             */
            case org.ametys.plugins.explorer.ObservationConstants.EVENT_RESOURCE_CREATED :
            case org.ametys.plugins.explorer.ObservationConstants.EVENT_RESOURCE_UPDATED :
            case org.ametys.plugins.explorer.ObservationConstants.EVENT_RESOURCE_RENAMED :
                url = (String) activity.get("parentFolderUrl");
                break;

            /*
             * CALENDAR
             */
            case org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_EVENT_CREATED :
            case org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_EVENT_UPDATED :
                url = (String) activity.get("eventUrl");
                break;

            /*
             * THREAD
             */
            case ObservationConstants.EVENT_THREAD_CREATED :
            case ObservationConstants.EVENT_THREAD_COMMENTED :
                url = (String) activity.get("threadUrl");
                break;

            /*
             * MEMBER
             */
            case ObservationConstants.EVENT_MEMBER_ADDED :
                // no url available
                break;
            case ObservationConstants.EVENT_WALLCONTENT_ADDED :
                // no url available
                break;

            /**
             * WIKI
             */
            case ObservationConstants.EVENT_MINISITE_PAGE_CREATED :
            case ObservationConstants.EVENT_MINISITE_PAGE_UPDATED :
            case ObservationConstants.EVENT_MINISITE_PAGE_RENAMED :
            case ObservationConstants.EVENT_MINISITE_PAGE_DELETED :
                url = (String) activity.get("pageUrl");
                break;

            /**
             * TASK
             */
            case ObservationConstants.EVENT_TASK_CREATED :
            case ObservationConstants.EVENT_TASK_ASSIGNED :
            case ObservationConstants.EVENT_TASK_CLOSED_STATUS_CHANGED :
                url = (String) activity.get("taskUrl");
                break;
            default:
                break;
        }
        if (StringUtils.isBlank(url))
        {
            url = (String) activity.get("projectUrl");
        }
        return url;
    }

    /**
     * Generate a description for this activity
     * @param activity the activity to describe
     * @param lang the language to use
     * @return a String of the description
     */
    @SuppressWarnings("unchecked")
    protected String getActivityDescription(Map<String, Object> activity, String lang)
    {
        String description = null;
        
        String eventType = (String) activity.get("type");
        
        Integer amount = (Integer) activity.get("amount");
        if (amount == null)
        {
            amount = 1;
        }
        
        String i18nKey = null;
        Map<String, I18nizableTextParameter> parameters = new HashMap<>();
        parameters.put("author", new I18nizableText(((Map<String, String>) activity.get("author")).get("fullname")));
        parameters.put("project", new I18nizableText((String) activity.get("projectTitle")));
        parameters.put("nb", new I18nizableText(amount.toString()));
        
        switch (eventType)
        {

            /*
             * RESOURCES
             */
            case org.ametys.plugins.explorer.ObservationConstants.EVENT_RESOURCE_CREATED :
                parameters.put("folder", new I18nizableText((String) activity.get("parentFolder")));
                if (amount == 1)
                {
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_DOCUMENT_ADDED_DESC";
                }
                else
                {
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_DOCUMENT_ADDED_MULTI_DESC";
                }
                break;
            case org.ametys.plugins.explorer.ObservationConstants.EVENT_RESOURCE_UPDATED :
                parameters.put("folder", new I18nizableText((String) activity.get("parentFolder")));
                if (amount == 1)
                {
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_DOCUMENT_UPDATED_DESC";
                }
                else
                {
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_DOCUMENT_UPDATED_MULTI_DESC";
                }
                break;
            case org.ametys.plugins.explorer.ObservationConstants.EVENT_RESOURCE_RENAMED :
                parameters.put("folder", new I18nizableText((String) activity.get("parentFolder")));
                if (amount == 1)
                {
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_DOCUMENT_RENAMED_DESC";
                }
                else
                {
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_DOCUMENT_RENAMED_MULTI_DESC";
                }
                break;

            /*
             * CALENDAR
             */
            case org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_EVENT_CREATED :
                parameters.put("calendar", new I18nizableText((String) activity.get("calendarTitle")));
                if (amount == 1)
                {
                    parameters.put("event", new I18nizableText(getTitles(activity, "events", "eventTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_CALENDAR_ADDED_DESC";
                }
                else
                {
                    parameters.put("events", new I18nizableText(getTitles(activity, "events", "eventTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_CALENDAR_ADDED_MULTI_DESC";
                }
                break;
            case org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_EVENT_UPDATED :
                parameters.put("calendar", new I18nizableText((String) activity.get("calendarTitle")));
                if (amount == 1)
                {
                    parameters.put("event", new I18nizableText(getTitles(activity, "events", "eventTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_CALENDAR_UPDATED_DESC";
                }
                else
                {
                    parameters.put("events", new I18nizableText(getTitles(activity, "events", "eventTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_CALENDAR_UPDATED_MULTI_DESC";
                }
                break;

            /*
             * THREAD
             */
            case ObservationConstants.EVENT_THREAD_CREATED :
                if (amount == 1)
                {
                    parameters.put("thread", new I18nizableText(getTitles(activity, "threads", "threadTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_THREAD_CREATED_DESC";
                }
                else
                {
                    parameters.put("threads", new I18nizableText(getTitles(activity, "threads", "threadTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_THREAD_CREATED_MULTI_DESC";
                }
                break;
            case ObservationConstants.EVENT_THREAD_COMMENTED :
                parameters.put("thread", new I18nizableText(getTitles(activity, "threads", "threadTitle")));
                i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_THREAD_POST_CREATED_DESC";
                break;

            /*
             * MEMBER
             */
            case ObservationConstants.EVENT_MEMBER_ADDED :
                if (amount == 1)
                {
                    parameters.put("member", new I18nizableText((String) activity.get("member")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_MEMBER_ADDED_DESC";
                }
                else
                {
                    Collection<Map<String, Object>> members = (Collection<Map<String, Object>>) activity.get("members");
                    List<String> names = members.stream().map(member -> (String) member.get("name")).collect(Collectors.toList());
                    parameters.put("members", new I18nizableText(String.join(", ", names)));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_MEMBERS_ADDED_DESC";
                }
                break;
                
            /**
             * WIKI
             */
            case ObservationConstants.EVENT_MINISITE_PAGE_CREATED :
                if (amount == 1)
                {
                    parameters.put("page", new I18nizableText(getTitles(activity, "pages", "pageTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_PAGE_CREATED_DESC";
                }
                else
                {
                    parameters.put("pages", new I18nizableText(getTitles(activity, "pages", "pageTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_PAGES_CREATED_DESC";
                }
                break;
            case ObservationConstants.EVENT_MINISITE_PAGE_UPDATED :
                if (amount == 1)
                {
                    parameters.put("page", new I18nizableText(getTitles(activity, "pages", "pageTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_PAGE_UPDATED_DESC";
                }
                else
                {
                    parameters.put("pages", new I18nizableText(getTitles(activity, "pages", "pageTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_PAGES_UPDATED_DESC";
                }
                break;
            case ObservationConstants.EVENT_MINISITE_PAGE_RENAMED :
                parameters.put("oldTitle", new I18nizableText((String) activity.get("pageOldTitle")));
                parameters.put("title", new I18nizableText((String) activity.get("pageTitle")));
                i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_PAGE_RENAMED_DESC";
                break;
            case ObservationConstants.EVENT_MINISITE_PAGE_DELETED :
                if (amount == 1)
                {
                    parameters.put("page", new I18nizableText(getTitles(activity, "pages", "pageTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_PAGE_DELETED_DESC";
                }
                else
                {
                    parameters.put("pages", new I18nizableText(getTitles(activity, "pages", "pageTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_PAGES_DELETED_DESC";
                }
                break;
                

            /**
             * NEWS
             */
            case ObservationConstants.EVENT_PROJECT_NEWS_PUBLISHED:
                parameters.put("contentTitle", new I18nizableText((String) activity.get("contentTitle")));
                i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_PROJECT_NEWS_ADDED_DESC";
                break;

            /**
             * WALLCONTENT
             */
            case ObservationConstants.EVENT_WALLCONTENT_ADDED :
                parameters.put("contentSummary", new I18nizableText((String) activity.get("contentSummary")));
                i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_WALLCONTENT_ADDED_WITH_TITLE_DESC";
                break;
                
            /**
             * NEWS AND WALLCONTENTS COMMENTS
             */
            case org.ametys.cms.ObservationConstants.EVENT_CONTENT_COMMENT_VALIDATED :
                parameters.put("contentTitle", new I18nizableText((String) activity.get("contentTitle")));
                i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_CONTENT_COMMENT_VALIDATED_DESC";
                break;

            /**
             * TASK
             */
            case ObservationConstants.EVENT_TASK_CREATED :
                if (amount == 1)
                {
                    parameters.put("task", new I18nizableText(getTitles(activity, "tasks", "taskTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_TASK_ADDED_DESC";
                }
                else
                {
                    parameters.put("tasks", new I18nizableText(getTitles(activity, "tasks", "taskTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_TASKS_ADDED_DESC";
                }
                break;
            case ObservationConstants.EVENT_TASK_DELETING :
                if (amount == 1)
                {
                    parameters.put("task", new I18nizableText(getTitles(activity, "tasks", "taskTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_TASK_DELETED_DESC";
                }
                else
                {
                    parameters.put("tasks", new I18nizableText(getTitles(activity, "tasks", "taskTitle")));
                    i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_TASKS_DELETED_DESC";
                }
                break;
            case ObservationConstants.EVENT_TASK_ASSIGNED :
                parameters.put("task", new I18nizableText((String) activity.get("taskTitle")));
                parameters.put("assignee", new I18nizableText((String) activity.get("assignees")));
                i18nKey = "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_TASK_ASSIGNED_DESC";
                break;
            case ObservationConstants.EVENT_TASK_CLOSED_STATUS_CHANGED :
                parameters.put("task", new I18nizableText((String) activity.get("taskTitle")));
                
                boolean isClosed = "true".equals(activity.get("isClosed"));
                i18nKey = isClosed ? "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_TASK_CLOSED_STATUS_CLOSE_DESC" : "PLUGINS_WORKSPACES_PROJECT_SERVICE_ACTIVITY_STREAM_EVENT_TASK_CLOSED_STATUS_OPEN_DESC";
                break;
            default:
                break;
        }
        
        if (i18nKey != null)
        {
            I18nizableText descriptionI18n = new I18nizableText("plugin.workspaces", i18nKey, parameters);
            description = _i18nUtils.translate(descriptionI18n, lang);
        }
        
        return description;
    }
    
    /**
     * Aggregates a list of elements from the activity.
     * activity = {
     *      mapKey : [
     *          {
     *              titleKey : "Title 1",
     *              …
     *          },
     *          {
     *              titleKey : "Title 2",
     *              …
     *          },
     *          {
     *              titleKey : "Title 3",
     *              …
     *          }
     *      ],
     *      …
     *  }
     * @param activity the activity to use
     * @param mapKey the key of the list to use
     * @param titleKey title to use from the map
     * @return a list, separated with commas, between each titles
     */
    protected String getTitles(Map<String, Object> activity, String mapKey, String titleKey)
    {
        @SuppressWarnings("unchecked")
        List<Map<String, String>> items = (List<Map<String, String>>) activity.get(mapKey);
        
        
        if (items == null)
        {
            return (String) activity.get(titleKey);
        }
        else
        {
            String result = null;
            
            for (Map<String, String> item : items)
            {
                if (result == null)
                {
                    result = item.get(titleKey);
                }
                else
                {
                    result += ", " + item.get(titleKey);
                }
            }
            
            return result;
        }
    }

}
