/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.action;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * An action that requires to be logged in
 */
abstract class AbstractLoggedAction extends AbstractPostAction
{
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    @Override
    public Map<String, Object> doAction(Request request, Map<String, Object> jsonParams)
    {
        Map<String, Object> result = new HashMap<>();
        
        if (_currentUserProvider.getUser() != null)
        {
            result = doLoggedInAction(request, jsonParams);
            
            // If no code is set, code is 200 (OK)
            if (!result.containsKey("code"))
            {
                result.put("code", 200);
            }
        }
        else
        {
            result.put("code", 403);
            throw new AccessDeniedException();
        }
        
        return result;
    }
    
    /**
     * Execute the action that requires to be logged in
     * @param request the request
     * @param jsonParams map of parameters passed in the post body as json
     * @return the json map to return
     */
    protected abstract Map<String, Object> doLoggedInAction(Request request, Map<String, Object> jsonParams);
}
