/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp;

import java.io.Serializable;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.quartz.SchedulerException;

import org.ametys.core.schedule.Runnable;
import org.ametys.core.schedule.Runnable.FireProcess;
import org.ametys.core.schedule.Runnable.MisfirePolicy;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.schedule.DefaultRunnable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import io.github.jav.exposerversdk.ExpoPushError;
import io.github.jav.exposerversdk.ExpoPushMessage;
import io.github.jav.exposerversdk.ExpoPushReceipt;
import io.github.jav.exposerversdk.ExpoPushTicket;
import io.github.jav.exposerversdk.PushClient;
import io.github.jav.exposerversdk.PushClientCustomData;
import io.github.jav.exposerversdk.PushClientException;
import io.github.jav.exposerversdk.PushNotificationErrorsException;
import io.github.jav.exposerversdk.PushNotificationException;
import io.github.jav.exposerversdk.enums.ReceiptError;
import io.github.jav.exposerversdk.enums.TicketError;

/**
 * Manager to push notifications to users
 */
public class PushNotificationManager extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = PushNotificationManager.class.getName();

    private UserPreferencesHelper _userPreferencesHelper;
    private Scheduler _scheduler;
    private CurrentUserProvider _currentUserProvider;

    public void service(ServiceManager manager) throws ServiceException
    {
        _userPreferencesHelper = (UserPreferencesHelper) manager.lookup(UserPreferencesHelper.ROLE);
        _scheduler = (Scheduler) manager.lookup(Scheduler.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    /**
     * Push a notification to all the registered devices of the users
     * @param title title of the message
     * @param message message to send
     * @param users list users to notify
     * @param data the data to push
     * @throws PushClientException something went wrong creating the push client
     * @throws SchedulerException an exception occurs setting up the scheduler for checking push receipts
     */
    public void pushNotifications(String title, String message, List<UserIdentity> users, Map<String, Object> data) throws PushClientException, SchedulerException
    {
        Map<UserIdentity, Set<String>> tokens = new HashMap<>();
        for (UserIdentity user : users)
        {
            tokens.put(user, _userPreferencesHelper.getNotificationTokens(user));
        }
        
        pushNotifications(title, message, tokens, data);
    }
    
    /**
     * Push a notification to a list of registered devices
     * @param title title of the message
     * @param message message to send
     * @param tokens list of device tokens
     * @param data the data to push
     * @throws PushClientException something went wrong creating the push client
     * @throws SchedulerException an exception occurs setting up the scheduler for checking push receipts
     */
    public void pushNotifications(String title, String message, Map<UserIdentity, Set<String>> tokens, Map<String, Object> data) throws PushClientException, SchedulerException
    {
        List<String> validTokens = tokens.values()
                                         .stream()
                                         .flatMap(Set::stream)
                                         .filter(PushClientCustomData::isExponentPushToken)
                                         .toList();

        if (validTokens.isEmpty())
        {
            getLogger().debug("No valid token found, aborting push notification for message '{}'", title);
            return;
        }
        
        // Invert tokens map, so we can easily retrieve the user from the token
        Map<String, UserIdentity> tokenToUserMap = new HashMap<>();
        for (Entry<UserIdentity, Set<String>> entry : tokens.entrySet())
        {
            for (String token : entry.getValue())
            {
                tokenToUserMap.put(token, entry.getKey());
            }
        }
        
        PushClient client = new PushClient();
        
        List<TicketInfo> tickets = _pushNotifications(title, message, tokenToUserMap, data, client);
        
        Map<String, String> recipientByTicket = new HashMap<>();
        
        for (TicketInfo ticketInfo : tickets)
        {
            ExpoPushTicket ticket = ticketInfo.ticket();
            String recipient = ticketInfo.token();
            switch (ticket.getStatus())
            {
                case OK ->
                {
                    recipientByTicket.put(ticket.getId(), recipient);
                }
                case ERROR ->
                {
                    if (ticket.getDetails().getError() == TicketError.DEVICENOTREGISTERED)
                    {
                        String pushToken = (String) ticket.getDetails().getAdditionalProperties().get("expoPushToken");

                        // double check to be sure
                        if (recipient.equals(pushToken))
                        {
                            UserIdentity user = tokenToUserMap.get(pushToken);
                            if (user != null)
                            {
                                getLogger().info("Remove unregistered token {} for user {}", pushToken, user);
                                _userPreferencesHelper.removeNotificationToken(pushToken, user);
                            }
                            else
                            {
                                // should not happen
                                getLogger().warn("Received unregistered token {} but cannot find associated user", pushToken);
                            }
                        }
                    }
                    else
                    {
                        getLogger().error("Received push error '{}': ", ticket.getDetails().getError(), ticket.getMessage());
                    }
                }
                default ->
                {
                    throw new IllegalArgumentException("Unknown status for push ticket: " + ticket.getStatus());
                }
            }
        }
        
        getLogger().info("Got successfully sent push tickets: {}", recipientByTicket);
        
        // Expo work is done, but we have to check in a few minutes if messages have been actually delivered by Apple and Google
        
        Map<String, TokenInfo> tokenInfos = recipientByTicket.entrySet()
                                                             .stream()
                                                             .collect(Collectors.toMap(Entry::getKey,
                                                                                       e -> new TokenInfo(e.getValue(),
                                                                                                          UserIdentity.userIdentityToString(tokenToUserMap.get(e.getValue())))));
        
        ZonedDateTime dateTime = ZonedDateTime.now().plusMinutes(15); // Expo recommends to check receipts after 15 minutes
        Runnable runnable = new DefaultRunnable(getClass().getName() + "$" + UUID.randomUUID(),
                                                new I18nizableText("plugin.mobileapp", "PLUGINS_MOBILEAPP_PUSH_RUNNABLE_LABEL"),
                                                new I18nizableText("plugin.mobileapp", "PLUGINS_MOBILEAPP_PUSH_RUNNABLE_DESCRIPTION"),
                                                FireProcess.CRON,
                                                dateTime.getSecond() + " " + dateTime.getMinute() + " " + dateTime.getHour() + " " + dateTime.getDayOfMonth() + " " + dateTime.getMonthValue() + " ? " + dateTime.getYear(),
                                                "org.ametys.plugins.mobileapp.push.schedule.PushReceipts",
                                                true /* removable */,
                                                false /* modifiable */,
                                                false /* deactivatable */,
                                                MisfirePolicy.FIRE_ONCE,
                                                true /* isVolatile */, // useless to persist it, expo tickets are themselves volatile
                                                _currentUserProvider.getUser(),
                                                Map.of("tickets", tokenInfos));

        _scheduler.scheduleJob(runnable);
    }
    
    @SuppressWarnings("unchecked")
    private List<TicketInfo> _pushNotifications(String title, String message, Map<String, UserIdentity> tokens, Map<String, Object> data, PushClient client)
    {
        getLogger().info("Pushing message '{}' with tokens {}", title, tokens);

        // Build the expo push message
        ExpoPushMessage expoPushMessage = new ExpoPushMessage();
        expoPushMessage.addAllTo(List.copyOf(tokens.keySet()));
        expoPushMessage.setTitle(title);
        expoPushMessage.setBody(message);

        expoPushMessage.setData(data);

        // Chunk the messages, each chunk will be sent in a single request
        List<List<ExpoPushMessage>> chunks = client.chunkPushNotifications(List.of(expoPushMessage));

        List<TicketInfo> allTickets = new ArrayList<>();
        for (List<ExpoPushMessage> chunk : chunks)
        {
            try
            {
                // Wait for each completable future to finish
                List<ExpoPushTicket> tickets = client.sendPushNotificationsAsync(chunk).join();
                
                // At this point, we should have as many recipients than received tickets
                List<String> recipients = chunk.stream()
                                               .map(ExpoPushMessage::getTo)
                                               .flatMap(List::stream)
                                               .toList();
                
                if (tickets.size() == recipients.size())
                {
                    for (int i = 0; i < tickets.size(); i++)
                    {
                        allTickets.add(new TicketInfo(recipients.get(i), tickets.get(i)));
                    }
                }
                else
                {
                    getLogger().warn("Number of push tickets {} is different from number of recipients {}, something went wrong", tickets.size(), recipients.size());
                }
            }
            catch (PushNotificationException e)
            {
                if (e.exception instanceof PushNotificationErrorsException ex)
                {
                    for (ExpoPushError error : ex.errors)
                    {
                        if ("PUSH_TOO_MANY_EXPERIENCE_IDS".equals(error.getCode()))
                        {
                            StringBuilder errorMessage = new StringBuilder("Push message " + title + " failed because tokens are associated with more than one expo project. Message will be split and sent again but you may want to remove wrong tokens:");
                            
                            List<Map<String, UserIdentity>> tokensByProject = new ArrayList<>();
                            
                            Map<String, Object> details = (Map<String, Object>) error.getAdditionalProperties().get("details");
                            for (String projectSlug : details.keySet())
                            {
                                errorMessage.append("\n\nFollowing tokens are associated with project ").append(projectSlug).append(':');
                                List<String> expoTokens = (List<String>) details.get(projectSlug);
                                
                                Map<String, UserIdentity> tokenForProject = new HashMap<>();
                                tokensByProject.add(tokenForProject);
                                for (String expoToken : expoTokens)
                                {
                                    UserIdentity userIdentity = tokens.get(expoToken);
                                    errorMessage.append('\n').append(expoToken).append(" for user ").append(UserIdentity.userIdentityToString(userIdentity));
                                    tokenForProject.put(expoToken, userIdentity);
                                }
                            }
                            
                            getLogger().error(errorMessage.toString());
                            
                            for (Map<String, UserIdentity> tokenForProject : tokensByProject)
                            {
                                allTickets.addAll(_pushNotifications(title, message, tokenForProject, data, client));
                            }
                        }
                        else
                        {
                            getLogger().error("Push notification {} returned an error with code '{}': {}", title, error.getCode(), error.getMessage(), ex);
                        }
                    }
                }
                else
                {
                    getLogger().error("Exception while sending push notification {}", title, e);
                }
            }
        }
        
        return allTickets;
    }
    
    /**
     * Check the status of previously sent push notifications
     * @param tickets the map of ticket id to token info (token + user)
     * @throws Exception something went wrong while checking the tickets
     */
    public void checkTickets(Map<String, TokenInfo> tickets) throws Exception
    {
        if (tickets.isEmpty())
        {
            getLogger().debug("No ticket to check");
            return;
        }
        
        getLogger().info("Checking tickets {}", tickets);
        
        PushClient client = new PushClient();
        
        // Chunk the messages, each chunk will be sent in a single request
        List<List<String>> chunks = client.chunkPushNotificationReceiptIds(List.copyOf(tickets.keySet()));
        
        for (List<String> chunk : chunks)
        {
            try
            {
                List<ExpoPushReceipt> receipts = client.getPushNotificationReceiptsAsync(chunk).join();
                
                for (ExpoPushReceipt receipt : receipts)
                {
                    if (receipt.getStatus() == io.github.jav.exposerversdk.enums.Status.ERROR && receipt.getDetails().getError() == ReceiptError.DEVICENOTREGISTERED)
                    {
                        TokenInfo tokenInfo = tickets.get(receipt.getId());
                        
                        if (tokenInfo != null)
                        {
                            String user = tokenInfo.user();
                            String token = tokenInfo.token();
                            
                            getLogger().warn("Device not registered for token {} associated with user {}, it will be removed", token, user);
                            
                            UserIdentity userIdentity = UserIdentity.stringToUserIdentity(user);
                            _userPreferencesHelper.removeNotificationToken(token, userIdentity);
                        }
                    }
                }
            }
            catch (PushNotificationException e)
            {
                getLogger().error("Exception while getting push receipts", e);
            }
        }
    }
    
    private record TicketInfo(String token, ExpoPushTicket ticket) { /* empty */ }
    
    /**
     * Association between an expo token and its associated user
     * @param token the Expo token
     * @param user the user identity as a string to be serializable
     */
    public record TokenInfo(String token, String user) implements Serializable { /* empty */ }
}
