/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp;

import java.time.LocalDate;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.collections4.CollectionUtils;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.site.Site;

/**
 * Helper to store/retreive conf per user
 */
public class UserPreferencesHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** The avalon role. */
    public static final String ROLE = UserPreferencesHelper.class.getName();

    private static final String __USERPREF_KEY_LANG = "lang";
    private static final String __USERPREF_KEY_SITE = "site";
    private static final String __USERPREF_KEY_ENABLED = "enabled";
    private static final String __USERPREF_KEY_ALL_FEEDS = "allFeeds";
    private static final String __USERPREF_KEY_FEEDS = "feeds";
    private static final String __USERPREF_KEY_ALL_PROJECTS = "allProjects";
    private static final String __USERPREF_KEY_PROJECTS = "projects";
    private static final String __USERPREF_KEY_ALL_TYPES = "allTypes";
    private static final String __USERPREF_KEY_TYPES = "types";

    /** User Preferences Manager */
    protected UserPreferencesManager _userPreferencesManager;

    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** JSON Utils */
    protected JSONUtils _jsonUtils;


    public void service(ServiceManager manager) throws ServiceException
    {
        _userPreferencesManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }

    /**
     * Get the list of impacted tokens for each feeds, for a user
     * @param user user to check
     * @param feedIds list of feeds to check
     * @param site the site for the validated content
     * @return a map with a Set of tokens for each feedId
     */
    @SuppressWarnings("unchecked")
    public Map<String, Set<String>> getUserImpactedTokens(UserIdentity user, Set<String> feedIds, Site site)
    {
        Map<String, Set<String>> feedsAndTokens = new HashMap<>();
        try
        {
            Map<String, String> unTypedUserPrefs = _userPreferencesManager.getUnTypedUserPrefs(user, "/mobileapp", Collections.emptyMap());
            for (Map.Entry<String, String> entry : unTypedUserPrefs.entrySet())
            {
                String token = entry.getKey();
                String value = entry.getValue();
                Map<String, Object> prefs = _jsonUtils.convertJsonToMap(value);
                
                List<String> feeds = (List<String>) prefs.get(__USERPREF_KEY_FEEDS);
                String siteName = (String) prefs.get(__USERPREF_KEY_SITE);
                
                boolean enabled = _getValueOrDefault((Boolean) prefs.get(__USERPREF_KEY_ENABLED), true);
                boolean allFeeds = _getValueOrDefault((Boolean) prefs.get(__USERPREF_KEY_ALL_FEEDS), feeds == null);
                
                if (enabled)
                {
                    boolean hasSiteQueries = site.hasValue(QueriesHelper.QUERY_CONTAINER_SITE_CONF_ID);
                    
                    if (siteName == null || hasSiteQueries && siteName.equals(site.getName()))
                    {
                        Collection<String> matchingFeeds = allFeeds || feeds == null ? feedIds : CollectionUtils.intersection(feeds, feedIds);
                        
                        for (String feedId : matchingFeeds)
                        {
                            Set tokens = feedsAndTokens.computeIfAbsent(feedId, __ -> new HashSet<>());
                            tokens.add(token);
                        }
                    }
                }
            }
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("Impossible to read a user notification token, for user '" + UserIdentity.userIdentityToString(user) + "'", e);
        }

        return feedsAndTokens;
    }

    /**
     * Get the list of impacted tokens for each feeds, for a user
     * @param user user to check
     * @param project test if a notification should be sent for this project
     * @param eventType test if a notification should be sent for this event type
     * @return a map with a the set of tokens impacted for each languages
     */
    @SuppressWarnings("unchecked")
    public Map<String, Set<String>> getUserImpactedTokens(UserIdentity user, Project project, String eventType)
    {
        Map<String, Set<String>> tokensForLanguage = new HashMap<>();
        try
        {
            Map<String, String> unTypedUserPrefs = _userPreferencesManager.getUnTypedUserPrefs(user, "/mobileapp", Collections.emptyMap());
            for (Map.Entry<String, String> entry : unTypedUserPrefs.entrySet())
            {
                String token = entry.getKey();
                String value = entry.getValue();
                Map<String, Object> prefs = _jsonUtils.convertJsonToMap(value);
                
                List<String> projects = (List<String>) prefs.get(__USERPREF_KEY_PROJECTS);
                List<String> types = (List<String>) prefs.get(__USERPREF_KEY_TYPES);
                
                boolean enabled = _getValueOrDefault((Boolean) prefs.get(__USERPREF_KEY_ENABLED), true);
                boolean allProjects = _getValueOrDefault((Boolean) prefs.get(__USERPREF_KEY_ALL_PROJECTS), projects == null);
                boolean allTypes = _getValueOrDefault((Boolean) prefs.get(__USERPREF_KEY_ALL_TYPES), types == null);
                
                if (enabled)
                {
                    boolean projectMatch = allProjects || projects == null || projects.contains(project.getId());
                    boolean typeMatch = allTypes || types == null || types.contains(eventType);
                    
                    if (projectMatch && typeMatch)
                    {
                        String lang = (String) prefs.get(__USERPREF_KEY_LANG);
                        
                        Set<String> tokens = tokensForLanguage.computeIfAbsent(lang, __ -> new HashSet<>());
                        tokens.add(token);
                    }
                }
            }
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("Impossible to read a user notification token, for user '" + UserIdentity.userIdentityToString(user) + "'", e);
        }

        return tokensForLanguage;
    }

    private boolean _getValueOrDefault(Boolean value, boolean defaultValue)
    {
        return value != null ? value : defaultValue;
    }

    /**
     * Get all notification tokens for current user
     * @return the list of notification tokens
     */
    public Set<String> getNotificationTokens()
    {
        UserIdentity user = _currentUserProvider.getUser();
        return getNotificationTokens(user);
    }

    /**
     * Get all notification tokens a user
     * @param user the user impacted
     * @return the list of notification tokens
     */
    public Set<String> getNotificationTokens(UserIdentity user)
    {
        try
        {
            Map<String, String> unTypedUserPrefs = _userPreferencesManager.getUnTypedUserPrefs(user, "/mobileapp", Collections.emptyMap());
            return unTypedUserPrefs.keySet();
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("Impossible to read a user notification token, for user '" + UserIdentity.userIdentityToString(user) + "'", e);
        }

        return Collections.EMPTY_SET;
    }

    /**
     * Removes a notification token for the current user
     * @param pushToken the token to remove
     */
    public void removeNotificationToken(String pushToken)
    {
        UserIdentity user = _currentUserProvider.getUser();
        removeNotificationToken(pushToken, user);
    }

    /**
     * Removes a notification token for a user
     * @param pushToken the token to remove
     * @param user the user impacted
     */
    public void removeNotificationToken(String pushToken, UserIdentity user)
    {
        try
        {
            Map<String, String> unTypedUserPrefs = _userPreferencesManager.getUnTypedUserPrefs(user, "/mobileapp", Collections.emptyMap());
            unTypedUserPrefs.remove(pushToken);
            _userPreferencesManager.setUserPreferences(user, "/mobileapp", Collections.emptyMap(), unTypedUserPrefs);
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("Impossible to remove a user notification token, for user '" + UserIdentity.userIdentityToString(user) + "'", e);
        }
    }

    /**
     * Remove all the notification tokens for a user
     * @param user the user impacted
     */
    public void removeAllNotificationTokens(UserIdentity user)
    {
        try
        {
            _userPreferencesManager.removeAllUserPreferences(user, "/mobileapp", Collections.emptyMap());
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("Impossible to remove all user notification tokens, for user '" + UserIdentity.userIdentityToString(user) + "'", e);
        }
    }

    /**
     * Save the notification settings for the current user
     * @param pushToken the token to impact
     * @param enabled if notifications are enabled
     * @param allFeeds if all feeds are allowed to be notified
     * @param feeds list of feeds ID for notifications
     * @param allProjects if all projects are allowed to be notified
     * @param projects list of project ID for notifications
     * @param allTypes if all types are allowed to be notified
     * @param types list of notifications types for notifications in projects
     * @param lang lang of this device
     */
    public void setNotificationSettings(String pushToken, boolean enabled, boolean allFeeds, List<String> feeds, boolean allProjects, List<String> projects, boolean allTypes, List<String> types, String lang)
    {
        UserIdentity user = _currentUserProvider.getUser();
        setNotificationSettings(pushToken, enabled, allFeeds, feeds, allProjects, projects, allTypes, types, lang, user);
    }

    /**
     * Save the notification settings for a user
     * @param pushToken token impacted by this settings
     * @param enabled if notifications are enabled
     * @param allFeeds if all feeds are allowed to be notified
     * @param feeds list of feeds ID for notifications
     * @param allProjects if all projects are allowed to be notified
     * @param projects list of project ID for notifications
     * @param allTypes if all types are allowed to be notified
     * @param types list of notifications types for notifications in projects
     * @param lang lang of this device
     * @param user the user impacted
     */
    public synchronized void setNotificationSettings(String pushToken, boolean enabled, boolean allFeeds, List<String> feeds, boolean allProjects, List<String> projects, boolean allTypes, List<String> types, String lang, UserIdentity user)
    {
        // synchronized because it happens that the app send multiple calls nearly simultaneously
        Map<String, Object> values = new HashMap<>();
        values.put(__USERPREF_KEY_LANG, lang);
        
        values.put(__USERPREF_KEY_ENABLED, enabled);
        
        values.put(__USERPREF_KEY_ALL_FEEDS, allFeeds);
        if (feeds != null)
        {
            values.put(__USERPREF_KEY_FEEDS, feeds);
        }
        
        values.put(__USERPREF_KEY_ALL_PROJECTS, allProjects);
        if (projects != null)
        {
            values.put(__USERPREF_KEY_PROJECTS, projects);
        }

        values.put(__USERPREF_KEY_ALL_TYPES, allTypes);
        if (types != null)
        {
            values.put(__USERPREF_KEY_TYPES, types);
        }

        values.put("epochDay", LocalDate.now().toEpochDay());

        String valuesAsString = _jsonUtils.convertObjectToJson(values);

        try
        {
            Map<String, String> unTypedUserPrefs = _userPreferencesManager.getUnTypedUserPrefs(user, "/mobileapp", Collections.emptyMap());
            if (unTypedUserPrefs == null)
            {
                unTypedUserPrefs = new HashMap<>();
            }

            unTypedUserPrefs.put(pushToken, valuesAsString);

            _userPreferencesManager.setUserPreferences(user, "/mobileapp", Collections.emptyMap(), unTypedUserPrefs);
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("Impossible to set user preferences for user '" + UserIdentity.userIdentityToString(user) + "'", e);
        }
    }

    /**
     * Get the notification settings for a token
     * @param pushToken the token to read
     * @param user the user impacted
     * @return a map containing feeds, projects and types as Set&lt;String&gt;, and epochDay as the last modification date
     */
    public Map<String, Object> getNotificationSettings(String pushToken, UserIdentity user)
    {
        try
        {
            Map<String, String> unTypedUserPrefs = _userPreferencesManager.getUnTypedUserPrefs(user, "/mobileapp", Collections.emptyMap());
            if (unTypedUserPrefs != null && unTypedUserPrefs.containsKey(pushToken))
            {
                String prefs = unTypedUserPrefs.get(pushToken);
                return _jsonUtils.convertJsonToMap(prefs);
            }
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("Impossible to retreive user preferences for user '" + UserIdentity.userIdentityToString(user) + "'", e);
        }
        return null;
    }

}
