/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.action;

import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;

import org.ametys.plugins.workspaces.project.ProjectsCatalogueManager;
import org.ametys.runtime.plugin.PluginsManager;

/**
 * Returns the list of projects for a user
 */
public class GetProjectsAction extends AbstractLoggedAction
{
    /** The project catalogue manager component */
    private ProjectsCatalogueManager _projectsCatalogueManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        if (PluginsManager.getInstance().isPluginActive("workspaces"))
        {
            _projectsCatalogueManager = (ProjectsCatalogueManager) smanager.lookup(ProjectsCatalogueManager.ROLE);
        }
    }
    
    @Override
    protected Map<String, Object> doLoggedInAction(Request request, Map<String, Object> jsonParams)
    {
        Map<String, Object> result = new HashMap<>();

        if (_projectsCatalogueManager != null)
        {
            List<Map<String, Object>> userProjects = _projectsCatalogueManager.getUserProjects();
            
            List<Map<String, Object>> feedList = userProjects.stream()
                .map(up -> _mapResult(up))
                .sorted(Comparator.comparing(project -> ((String) project.get("name")).toLowerCase()))
                .toList();
            result.put("feed_list", feedList);
        }
        
        return result;
    }
    
    /**
     * Map a json used for project list to a map used by the mobile app
     * @param userProject a map representing a project
     * @return a map used by the mobile app
     */
    private Map<String, Object> _mapResult(Map<String, Object> userProject)
    {
        Map<String, Object> result = new HashMap<>();

        result.put("feed_id", userProject.get("id"));
        result.put("name", userProject.get("title"));
        result.put("url", userProject.get("url"));
        result.put("illustration", userProject.get("illustration"));
        result.put("language", userProject.get("language"));
        if (userProject.containsKey("category"))
        {
            result.put("color", ((Map) userProject.get("category")).get("color"));
        }
        
        return result;
    }
}
