/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.action;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.environment.http.HttpEnvironment;
import org.apache.cocoon.environment.http.HttpResponse;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.util.JSONUtils;

/**
 * An abstract action to bet the request parameters (esther in GET or POST)
 */
abstract class AbstractPostAction extends ServiceableAction
{
    /** JSON Utils */
    protected JSONUtils _jsonUtils;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
    }
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Map<String, Object> jsonParams = getParameters(objectModel);

        Map<String, Object> result = doAction(request, jsonParams);

        request.setAttribute(JSonReader.OBJECT_TO_READ, result);

        if (result.get("code") != Integer.valueOf(200))
        {
            HttpResponse response = (HttpResponse) ObjectModelHelper.getResponse(objectModel);
            response.setStatus((int) result.get("code"));
        }
        return EMPTY_MAP;
    }

    /**
     * Get a map of parameters from the POST body
     * @param objectModel the objectModel from whidh the body will be read
     * @return a Map representing the body, parsed as json
     * @throws IOException something went wrong
     */
    protected Map<String, Object> getParameters(Map objectModel) throws IOException
    {
        HttpServletRequest postReq = (HttpServletRequest) objectModel.get(HttpEnvironment.HTTP_REQUEST_OBJECT);

        InputStream postBody = postReq.getInputStream();
        String body = new String(postBody.readAllBytes(), StandardCharsets.UTF_8);
        Map<String, Object> json = _jsonUtils.convertJsonToMap(body);

        return json;
    }

    /**
     * Get a parameter either from the POST body or directly from the request parameters
     * @param name the name of the parameter to fetch
     * @param jsonParams the json parameters passed
     * @param request the request to read from
     * @return an Object (if from json) or a String (if from the request)
     */
    protected Object getParameter(String name, Map<String, Object> jsonParams, Request request)
    {
        if (jsonParams.containsKey(name))
        {
            return jsonParams.get(name);
        }
        else if (request.getParameter(name) != null)
        {
            return request.getParameter(name);
        }
        else
        {
            return request.getAttribute(name);
        }
    }

    /**
     * Execute the action
     * @param request the request
     * @param jsonParams map of parameters passed in the post body as json
     * @return the json map to return
     */
    protected abstract Map<String, Object> doAction(Request request, Map<String, Object> jsonParams);
}
