/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.observer;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.RichTextHelper;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.right.AllowedUsers;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.mobileapp.PushNotificationManager;
import org.ametys.plugins.mobileapp.QueriesHelper;
import org.ametys.plugins.mobileapp.UserPreferencesHelper;
import org.ametys.plugins.queriesdirectory.Query;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.site.Site;

/**
 * On validation, test each query to notify impacted users
 */
public class ContentValidatedObserver extends AbstractLogEnabled implements AsyncObserver, Serviceable, Contextualizable
{
    private static final String __DESCRIPTION_MAX_SIZE_CONF_ID = "plugin.mobileapp.push.description.richtext.max";
    
    private Context _context;

    private QueriesHelper _queryHelper;
    private UserPreferencesHelper _userPreferencesHelper;
    private PushNotificationManager _pushNotificationManager;
    private UserManager _userManager;
    private UserPopulationDAO _userPopulationDAO;
    private RightManager _rightManager;
    private RichTextHelper _richTextHelper;
    private AmetysObjectResolver _resolver;

    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _queryHelper = (QueriesHelper) manager.lookup(QueriesHelper.ROLE);
        _userPreferencesHelper = (UserPreferencesHelper) manager.lookup(UserPreferencesHelper.ROLE);
        _pushNotificationManager = (PushNotificationManager) manager.lookup(PushNotificationManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _userPopulationDAO = (UserPopulationDAO) manager.lookup(UserPopulationDAO.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _richTextHelper = (RichTextHelper) manager.lookup(RichTextHelper.ROLE);
    }

    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_CONTENT_VALIDATED) 
            || event.getId().equals(ObservationConstants.EVENT_CONTENT_TAGGED);
    }

    public int getPriority()
    {
        return MIN_PRIORITY;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        // FIXME : Temporary fix to let time to the content to be visible from Solr queries
        Thread.sleep(5000);
        
        String contentId = (String) event.getArguments().get(ObservationConstants.ARGS_CONTENT_ID);
        
        Request request = ContextHelper.getRequest(_context);

        // Retrieve current workspace
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Use live workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, WebConstants.LIVE_WORKSPACE);
            
            if (!_resolver.hasAmetysObjectForId(contentId))
            {
                getLogger().debug("Content with id {} does not exists in workspace live, no push notification will be sent.", contentId);
                return;
            }
        
            Content content = _resolver.resolveById(contentId);
    
            if (!(content instanceof WebContent webContent))
            {
                getLogger().debug("We currently do not support push notifications for off-site content {}", content.getId());
                return;
            }
            
            // First find the queries that whose results contain the content
            Site site = webContent.getSite();
            Set<Query> queries = _queryHelper.getQueriesForResult(content.getId(), site);
    
            getLogger().info("{} queries found for content {}", queries.size(), content.getId());
            
            if (queries.isEmpty())
            {
                return;
            }
    
            // Then collect users that have read access to the content
            Set<UserIdentity> users;
            AllowedUsers readAccessAllowedUsers = _rightManager.getReadAccessAllowedUsers(content);
            if (readAccessAllowedUsers.isAnonymousAllowed() || readAccessAllowedUsers.isAnyConnectedUserAllowed())
            {
                List<String> userPopulationsIds = _userPopulationDAO.getUserPopulationsIds();
                Collection<User> allUsers = _userManager.getUsersByPopulationIds(userPopulationsIds);
                users = allUsers.stream().map(User::getIdentity).collect(Collectors.toSet());
            }
            else
            {
                users = readAccessAllowedUsers.resolveAllowedUsers(true);
            }
    
            Map<String, Query> queryMap = queries.stream().collect(Collectors.toMap(Query::getId, Function.identity()));
            Set<String> feedIds = queryMap.keySet();
    
            // Then find the users that have subscribed to the queries
            Map<String, Map<UserIdentity, Set<String>>> notificationsNeeded = new HashMap<>();
            for (UserIdentity user : users)
            {
                Map<String, Set<String>> tokensForUser = _userPreferencesHelper.getUserImpactedTokens(user, feedIds, site);
                for (Entry<String, Set<String>> tokensByFeed : tokensForUser.entrySet())
                {
                    Query query = queryMap.get(tokensByFeed.getKey());
                    
                    if (_rightManager.hasReadAccess(user, query))
                    {
                        Map<UserIdentity, Set<String>> tokens = notificationsNeeded.computeIfAbsent(tokensByFeed.getKey(), __ -> new HashMap<>());
                        tokens.put(user, tokensByFeed.getValue());
                    }
                }
            }
    
            Map<String, String> data = _queryHelper.getDataForContent(content);
            Map<String, String> sorts = queries.stream().collect(Collectors.toMap(Query::getId, q -> _queryHelper.getSortProperty(q, true).get(0).sortField()));
    
            // Finally send the notifications
            for (Entry<String, Map<UserIdentity, Set<String>>> entry : notificationsNeeded.entrySet())
            {
                Map<String, Object> notificationData = new HashMap<>();
                notificationData.putAll(data);
                String feedId = entry.getKey();
                notificationData.put("feed_id", feedId);
                if (queryMap.containsKey(feedId))
                {
                    notificationData.put("category_name", queryMap.get(feedId).getTitle());
                }
    
                String sortField = null;
                if (sorts.containsKey(feedId))
                {
                    sortField = sorts.get(feedId);
                }
                String isoDate = _queryHelper.getContentFormattedDate(content, sortField);
                notificationData.put("date", isoDate);
    
                String description = _getContentDescription(content);
    
                _pushNotificationManager.pushNotifications(content.getTitle(), description, entry.getValue(), notificationData);
            }
        }
        finally
        {
            // Restore context
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }

    /**
     * Get a description for the notification.
     * It will first try to read a "abstract" value in the content, and if not available, will try to read a rich-text stored in "content" (and cut it down).
     * If none is available, an empty String is returned.
     * @param content The content to read
     * @return a description for this content
     */
    protected String _getContentDescription(Content content)
    {
        Long maxSize = Config.getInstance().getValue(__DESCRIPTION_MAX_SIZE_CONF_ID);
        String result = "";

        if (content.hasValue("abstract") && org.ametys.runtime.model.type.ModelItemTypeConstants.STRING_TYPE_ID.equals(content.getDefinition("abstract").getType().getId()))
        {
            result = content.getValue("abstract", false, "");
        }
        else if (content.hasValue("content") && ModelItemTypeConstants.RICH_TEXT_ELEMENT_TYPE_ID.equals(content.getDefinition("content").getType().getId()))
        {
            RichText richText = content.getValue("content");
            result = _richTextHelper.richTextToString(richText, maxSize.intValue());
        }

        return result;
    }
}
